/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.security.auth.ws;

import static org.jboss.soa.esb.services.security.auth.ws.SoapExtractionUtil.isEndOfHeader;
import static org.jboss.soa.esb.services.security.auth.ws.SoapExtractionUtil.isStartOfBody;
import static org.jboss.soa.esb.services.security.auth.ws.SoapExtractionUtil.isStartOfHeader;
import java.io.StringReader;
import java.util.HashSet;
import java.util.Set;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.Characters;
import javax.xml.stream.events.EndElement;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequest;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequestImpl;
import org.jboss.soa.esb.services.security.auth.ExtractionException;
import org.jboss.soa.esb.services.security.auth.SecurityInfoExtractor;
import org.jboss.soa.esb.services.security.principals.User;

/**
 * WSSecurityInfoExtractor extracts security related information from a SOAP
 * security header. 
 * </p> 
 * If the security element contains a 'UsernameToken' element this will be extraced into 
 * a principal with the username and a credential being the password represented as a character array.
 * </p>
 * 
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 * 
 */
public class UsernameTokenExtractor implements SecurityInfoExtractor<String>
{
    private Logger log = Logger.getLogger(UsernameTokenExtractor.class);

    private static final XMLInputFactory XML_INPUT_FACTORY = getXmlInputFactory();

    private final QName usernameTokenQName;
    private final QName usernameQName;
    private final QName passwordQName;

    public UsernameTokenExtractor(final String securityNS)
    {
        AssertArgument.isNotNullAndNotEmpty(securityNS, "securityNS");
        
        usernameTokenQName = new QName(securityNS, "UsernameToken");
        usernameQName = new QName(securityNS, "Username");
        passwordQName = new QName(securityNS, "Password");
    }

    /**
     * Extracts UsernameToken element is one exists and creates a Principal with
     * the username and sets a Credential using the password. The type of the
     * Credential is a character array.
     * <p/>
     * If the SOAP message contains a BinarySecurityToken this will be extracted
     * and added as a Credential.
     * 
     * @param soap the soap message represented as a String
     * @return {@link AuthenticationRequest}
     */
    public AuthenticationRequest extractSecurityInfo(final String soap) throws ExtractionException
    {
        if (soap == null || !soap.startsWith("<"))
            return null;

        XMLEventReader xmlReader = null;
        User user = null;
        Set<Object> credentials = new HashSet<Object>();
        try
        {
            xmlReader = XML_INPUT_FACTORY.createXMLEventReader(new StringReader(soap));

            while (xmlReader.hasNext())
            {
                XMLEvent xmlEvent = xmlReader.nextEvent();
                if (isStartOfHeader(xmlEvent))
                {
                    while (xmlReader.hasNext())
                    {
                        xmlEvent = xmlReader.nextEvent();
		                if (isStartOfUsernameToken(xmlEvent))
		                {
		                    while (xmlReader.hasNext())
		                    {
		                        xmlEvent = xmlReader.nextEvent();
		                        if (isStartOfUsername(xmlEvent))
		                        {
		                            XMLEvent username = xmlReader.nextEvent();
		                            if (username.isCharacters())
		                            {
		                                // username is a required element.
		                                user = new User(((Characters) username).getData());
		                            }
		                        }
	                        
		                        if (isStartOfPassword(xmlEvent))
		                        {
		                            final XMLEvent password = xmlReader.nextEvent();
		                            if (password.isCharacters())
		                            {
		                                final Characters characters = (Characters) password;
		                                final String data = characters.getData();
		                                // Password is an optional element
		                                if (data != null)
			                                credentials.add(characters.getData());
		                            }
		                        }
				                if (isEndOfUsernameToken(xmlEvent))
				                {
							        if (credentials.isEmpty())
							        {
							            return null;
							        }
							        return new AuthenticationRequestImpl.Builder(user, credentials).build();
				                }
	                        }
	                    }
		                if (isEndOfHeader(xmlEvent))
		                {
		                    return null;
		                }
                    }
                }
                
                if (isStartOfBody(xmlEvent))
                    return null;
            }
        }
        catch (XMLStreamException e)
        {
            throw new ExtractionException(e.getMessage(), e);
        }
        finally
        {
            close(xmlReader);
        }

        return null;
    }
    
    private boolean isStartOfUsername(final XMLEvent event)
    {
        return event.isStartElement() && ((StartElement) event).getName().equals(usernameQName);
    }
    
    private boolean isStartOfPassword(final XMLEvent event)
    {
        return event.isStartElement() && ((StartElement) event).getName().equals(passwordQName);
    }
    
    private boolean isStartOfUsernameToken(final XMLEvent event)
    {
        return event.isStartElement() && ((StartElement) event).getName().equals(usernameTokenQName);
    }
    
    private boolean isEndOfUsernameToken(final XMLEvent event)
    {
        return event.isEndElement() && ((EndElement) event).getName().equals(usernameTokenQName);
    }
    
    private static XMLInputFactory getXmlInputFactory()
    {
        final XMLInputFactory factory = XMLInputFactory.newInstance();
        // set any properies here if required before returning.
        return factory;
    }

    private void close(final XMLEventReader reader)
    {
        if (reader != null)
        {
            try
            {
                reader.close();
            }
            catch (final XMLStreamException ignore)
            {
                log.error("XMLStreamException caught while trying to close the XMLEventReader", ignore);
            }
        }
    }

}
