package org.jboss.soa.esb.message;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

/**
 * The message body holds arbitrary information which represents the payload as
 * far as clients and services are concerned. A body may contain:
 * 
 * (i) a byte array for arbitrary data. How that array is interpreted by the
 * service is implementation specific and outside the scope of the ESB to
 * enforce.
 * 
 * (ii) a list of Objects of arbitrary types. How these objects are serialized
 * to/from the message body when it is transmitted is up to the specific Object
 * type. The plan is to add support for various TYPES of Object and the message
 * implementation will use external adapters to externalize/internalize the
 * Objects. Currently we only support Serializable objects.
 * 
 * Given that there are attachments, properties, byte arrays and named objects,
 * you may be wondering where should you put your payload? The answer is fairly
 * straightforward: 
 * 
 * As a service developer, you define the contract that clients
 * use in order to interact with your service. As part of that contract, you
 * will specific both functional and non-functional aspects of the service,
 * e.g., that it is an airline reservation service (functional) and that it is
 * transactional (non-functional). You'll also define the operations (messages)
 * that the service can understand. As part of the message definition, you
 * stipulate the format (e.g., Java Serialized message versus XML) and the
 * content (e.g., transaction context, seat number, customer name etc.) When
 * defining the content, you can specify where in the Message your service will
 * expect to find the payload. That can be in the form of attachments, specific
 * named objects (even the default named object if you so wish), or the byte
 * array. It is entirely up to the service developer to determine. The only
 * restrictions are that objects and attachments must be globally uniquely
 * named, or one service (or Action) may inadvertently pick up a partial payload
 * meant for another if the same Message Body is forwarded across multiple hops.
 * 
 * As a service users, you obtain the contract definition about the service
 * (e.g., through UDDI or out-of-band communication) and this will define where
 * in the message the payload must go. Information placed in other locations
 * will likely be ignored and result in incorrect operation of the service.
 * 
 * Note, in order to maintain loose coupling, services should not expose any backend
 * implementation choices. As such you should try to avoid sending Java Serialized
 * instances within the Body where possible because it does limit your future choices
 * on service implementation to Java and potentially specific versions of Java.
 */

public interface Body
{
	public static final String DEFAULT_LOCATION = "org.jboss.soa.esb.message.defaultEntry";

	/**
	 * Add the specified Object at the default location within the message. If
	 * the default location is already used then the contents will be
	 * overwritten.
	 * 
	 * @param value
	 */

	public void add(Object value);

	/**
	 * Add the specified Object to the body.
	 * 
	 * @param name
	 *            The name of the object. MUST be unique within this body. If null
	 *            then an exception will be thrown. This is to make sure that some
	 *            computational errors don't arbitrarily resolve to data within the message body that could
	 *            then be misinterpreted. If someone wants to go with the default name
	 *            then they can either use add(value) or explicitly use the default name.
	 * @param value
	 *            The Object to add.
	 */

	public void add(String name, Object value);

	/**
	 * Get the Object at the default location in the message, or
	 * <code>null</code> otherwise.
	 * 
	 * @return the object.
	 */

	public Object get();

	/**
	 * Get the specified Object, or <code>null</code> if not present.
	 * 
	 * @param name
	 *            the name of the Object to retrieve.  If null
	 *            then an exception will be thrown. This is to make sure that some
	 *            computational errors don't arbitrarily resolve to data within the message body that could
	 *            then be misinterpreted. If someone wants to go with the default name
	 *            then they can either use add(value) or explicitly use the default name.
	 * @return the Object.
	 */

	public Object get(String name);

	/**
	 * Remove the specified Object and return it, or <code>null</code> if not
	 * present.
	 * 
	 * @param name
	 *            the name of the Object to remove.
	 * @return the Object removed.
	 */

	public Object remove(String name);

	/**
	 * Set the byte content of the body. This does not affect any of the named
	 * objects or attachments.
	 * 
	 * @param content
	 *            the message bytes
	 * @deprecated As of 4.2 you can use add(BytesBody.BYTES_LOCATION, content)
	 */

	public void setContents(byte[] content);

	/**
	 * @return the byte content of the body.
	 * @deprecated As of 4.2 you can use get(BytesBody.BYTES_LOCATION)
	 */

	public byte[] getContents();

	/**
	 * Set the byte content of the body. This does not affect any of the named
	 * objects or attachments.
	 * 
	 * @param content
	 *            the message bytes
	 * @deprecated As of 4.2 you can use add(BytesBody.BYTES_LOCATION, content)
	 */

	public void setByteArray(byte[] content);

	/**
	 * @return the byte content of the body.
	 * @deprecated As of 4.2 you can use get(BytesBody.BYTES_LOCATION)
	 */

	public byte[] getByteArray();

	/**
	 * Replace this body instance with the one given.
	 * 
	 * This method is not thread safe.
	 * 
	 * @param b
	 *            the body to be replaced with.
	 */

	public void replace(Body b);

	/**
	 * Merge two bodies. Any duplicate entries in the current instance will be
	 * lost in favour of the new instance.
	 * 
	 * This method is not thread safe.
	 * 
	 * @param b
	 *            the body to be merged with.
	 */

	public void merge(Body b);

	/**
	 * @return get the list of names in this instance.
	 */

	public String[] getNames();

	// TODO replace an entry in the body

}