/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.message;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.util.BeanConfigurator;

/**
 * ActionBeanConfigurator was created because ActionMapper flattens the
 * ConfigTree by promoting child "property" elements to attributes on the parent
 * element.
 * 
 * @author <a href="mailto:rex.sheridan@sapience360.com">Rex Sheridan</a>
 * 
 */
public class ActionBeanConfigurator extends BeanConfigurator {

	/**
	 * A list of key-value pairs to ignore because they don't actually
	 * correspond to bean properties.
	 */
	private static final List<String> ignored = Arrays.asList("action",
			ListenerTagNames.ACTION_ELEMENT_TAG,
			ListenerTagNames.ACTION_CLASS_TAG,
			ListenerTagNames.LISTENER_CLASS_TAG,
			ListenerTagNames.PROCESS_METHOD_TAG);
	
	private static final Map<String, String> optionalFields = getOptionalFieldsMap();
	
	/**
	 * @param config
	 * @param bean
	 */
	ActionBeanConfigurator(ConfigTree config, Object bean) {
	    AssertArgument.isNotNull(config, "config");
	    
		setBean(bean);
		List<KeyValuePair> pairs = config.attributesAsList();
		
		ConfigTree parent = config.getParent();
		if (parent != null)  {
    		List<KeyValuePair> parentAttributes = parent.attributesAsList();
    		for (KeyValuePair attr : parentAttributes)  {
    		    String newKeyName = optionalFields.get(attr.getKey());
    		    if (newKeyName != null) {
    		        pairs.add(new KeyValuePair(newKeyName, attr.getValue()));
    		    }
            }
		}
		
		Map<String, String> properties = new HashMap<String, String>();
		for (KeyValuePair kvp : pairs) {
			if (!ignored.contains(kvp.getKey())) {
				properties.put(kvp.getKey(), kvp.getValue());
			}
		}
		ConfigTree[] children = config.getAllChildren();
		for (ConfigTree child : children) {
			properties.put(child.getName(), child.toXml());
		}
		setProperties(properties);
		setOptionalFields(optionalFields.values());
	}


    /**
     * Maps configuration tag names to java field names. Some of the configuration
     * names have '-' in them. This map contains the configuration names as its 
     * keys and the java field name as its value.
     * 
     * @return Map<String, String> Map of configuration names to java bean names(field names).
     */
    static Map<String, String> getOptionalFieldsMap()
    {
        Map<String, String> map = new HashMap<String, String>();
        map.put(ListenerTagNames.SERVICE_CATEGORY_NAME_TAG, "serviceCategory");
        map.put(ListenerTagNames.SERVICE_NAME_TAG, "serviceName");
        map.put(ListenerTagNames.SERVICE_DESCRIPTION_TAG, "serviceDescription");
        map.put(ListenerTagNames.MAX_THREADS_TAG, "maxThreads");
        map.put(ListenerTagNames.MEP_ATTRIBUTE_TAG, "mep");
        return map;
    }
	
}
