/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.gateway.mina;

import java.net.InetSocketAddress;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.util.ClassUtil;

/**
 * Deals with UDP configuration properties and parsing.
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 * 
 */
public class UdpGatewayConfig
{
    /**
     * Host name configuration attribute name.
     */
    public static final String HOST_ATTR = "host";

    /**
     * Port configuration attribute name.
     */
    public static final String PORT_ATTR = "port";

    /**
     * Handler class configuration attribute name.
     */
    public static final String HANDLER_CLASS_ATTR = "handlerClass";
    
    /**
     * ConfigTree with contains the elements and attributes from the
     * configuration file.
     */
    private final ConfigTree config;

    /**
     * Sole constructor.
     * 
     * @param config
     *            The {@link ConfigTree} that contains the configuration
     *            information.
     */
    public UdpGatewayConfig(final ConfigTree config)
    {
        AssertArgument.isNotNull(config, "config");
        this.config = config;
    }

    /**
     * Gets the port from the configuration.
     * 
     * @return {@code int} the port that the gateway should listen to.
     * 
     * @throws ConfigurationException
     *             If the port was missing from the configuration or it could
     *             not be parse as an int.
     */
    public int getPort() throws ConfigurationException
    {
        return parsePort(config.getRequiredAttribute(PORT_ATTR));
    }

    /**
     * Gets the {@link InetSocketAddress} that this gateway should listen to.
     * 
     * @return {@code InetSocketAddress} the {@link InetSocketAddress} that the
     *         gateway should listen to.
     * 
     * @throws ConfigurationException
     *             If the host or port was missing from the configuration or if
     *             the port could not be parsed.
     */
    public InetSocketAddress getSocketAddress() throws ConfigurationException
    {
        return new InetSocketAddress(config.getRequiredAttribute(HOST_ATTR), getPort());
    }
    
    /**
     * Gets the target service category from the configuration.
     * 
     * @return {@code String} The target service category
     * 
     * @throws ConfigurationException
     *             If service category attribute is missing from the
     *             configuration.
     */
    public String getServiceCategory() throws ConfigurationException
    {
        return config.getRequiredAttribute(ListenerTagNames.TARGET_SERVICE_CATEGORY_TAG);
    }

    /**
     * Gets the target service name from the configuration.
     * 
     * @return {@code String} The target service name
     * 
     * @throws ConfigurationException
     *             If service name attribute is missing from the configuration.
     */
    public String getServiceName() throws ConfigurationException
    {
        return config.getRequiredAttribute(ListenerTagNames.TARGET_SERVICE_NAME_TAG);
    }

    /**
     * Get the configured {@link MessageHandler} instance. <p/> Defaults to
     * {@link DefaultMessageHandler}.
     * 
     * @param config
     *            The listener configuration.
     * @return The MessageHandler implementation.
     * @throws ConfigurationException
     *             Unable to create the configured {@link MessageHandler}
     *             instance.
     */
    public MessageHandler getHandler() throws ConfigurationException
    {
        String handlerClass = config.getRequiredAttribute(HANDLER_CLASS_ATTR);
        try
        {
            return (MessageHandler) ClassUtil.forName(handlerClass, getClass()).newInstance();
        } 
        catch (final ClassCastException e)
        {
            throw new ConfigurationException("Class '" + handlerClass + "' must implement '" + MessageHandler.class.getName() + "'.");
        } 
        catch (final InstantiationException e)
        {
            throw new ConfigurationException("Failed to create an instance of MessagHandler implementation '" + handlerClass + "'.", e);
        } 
        catch (final IllegalAccessException e)
        {
            throw new ConfigurationException("Failed to create an instance of MessagHandler implementation '" + handlerClass + "'.", e);
        } 
        catch (final ClassNotFoundException e)
        {
            throw new ConfigurationException("Failed to create an instance of MessagHandler implementation '" + handlerClass + "'.", e);
        }
    }
    
    public static int parsePort(String portString) throws ConfigurationException
    {
        int parseInt;
        try
        {
            parseInt = Integer.parseInt(portString);
        } catch (final NumberFormatException e)
        {
            throw new ConfigurationException("port must be a postive integer. It was '" + portString + "'");
        }

        if (parseInt <= 0)
        {
            throw new ConfigurationException("port must be a postive integer. It was '" + portString + "'");
        }
        return parseInt;
    }

}
