/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.gateway;

import org.jboss.soa.esb.listeners.message.MessageComposer;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.listeners.ListenerUtil;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.eprs.FTPEpr;
import org.jboss.soa.esb.util.*;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.internal.soa.esb.assertion.AssertArgument;

import java.io.File;
import java.io.IOException;

/**
 * Remote File Message Composer.
 * <p/>
 * The logic in this class was lifeted from {@link RemoteGatewayListener}.
 *
 * @see org.jboss.soa.esb.listeners.gateway.LocalFileMessageComposer
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class RemoteFileMessageComposer<T extends File> implements MessageComposer<T> {
 
    private FTPEpr ftpEpr;
    private File downloadDir;
    private MessagePayloadProxy payloadProxy;

    public void setConfiguration(ConfigTree config) throws ConfigurationException {
        EPR epr = ListenerUtil.assembleEpr(config);

        if (!(epr instanceof FTPEpr))  {
            throw new ConfigurationException("This Gateway only accepts FTP and SFTP.");
        }
        ftpEpr = (FTPEpr) epr;
        // This may look a bit odd... The "file" input dir is our FTP download dir...
        downloadDir = AbstractFileGateway.getFileInputDirectory(config);
        payloadProxy = new MessagePayloadProxy(config,
                new String[] {BytesBody.BYTES_LOCATION},
                new String[] {BytesBody.BYTES_LOCATION});
    }

    public Message compose(T inputFile) throws MessageDeliverException {
        AssertArgument.isNotNull(inputFile, "inputFile");

        Message message = MessageFactory.getInstance().getMessage();
        try {
            payloadProxy.setPayload(message, getPayload(inputFile));
        } catch (IOException e) {
            throw new MessageDeliverException("Error reading remote input file '" + inputFile.getAbsolutePath() + "'.", e);
        } catch (RemoteFileSystemException e) {
            throw new MessageDeliverException("Error reading remote input file '" + inputFile.getAbsolutePath() + "'.", e);
        }

        return message;
    }
    
    protected Object getPayload( T inputFile ) throws RemoteFileSystemException, IOException  {
        return getFileContents(inputFile);
    }

    public Object decompose(Message message, T inputMessage) throws MessageDeliverException {
        return payloadProxy.getPayload(message);
    }

    private byte[] getFileContents(File file) throws IOException, RemoteFileSystemException {
        RemoteFileSystem remoteFileSystem = null;
        File temp = File.createTempFile("FTPdown", ".tmp");

        try
        {
            remoteFileSystem = RemoteFileSystemFactory.getRemoteFileSystem(ftpEpr, true);
            remoteFileSystem.setRemoteDir(FtpClientUtil.fileToFtpString(downloadDir));
            remoteFileSystem.downloadFile(file.toString(), temp.getAbsolutePath());

            return FileUtil.readFile(temp);
        } finally {
            temp.delete();
            if (null != remoteFileSystem) {
                remoteFileSystem.quit();
            }
        }
    }
}
