/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.soa.esb.listeners.gateway;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import javax.jms.BytesMessage;
import javax.jms.JMSException;
import javax.jms.ObjectMessage;
import javax.jms.TextMessage;
import org.apache.log4j.Logger;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.Body;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.helpers.ConfigTree;

/**
* Default gateway action for plain jms messages <p/>
* It will just drop the jms message contents into a esb Message
*
* @author <a href="mailto:schifest@heuristica.com.ar">schifest@heuristica.com.ar</a>
* @author <a href="mailto:daniel.bevenius@redpill.se">Daniel Bevenius</a>
* @since Version 4.0
*
*/
public class PackageJmsMessageContents
{
	private Logger log = Logger.getLogger( PackageJmsMessageContents.class );
	/**
	 * Strategy for setting JMS Properties on the ESB Message object created
	 * by the process method.
	 */
	private ESBPropertiesSetter esbPropertiesStrategy = new DefaultESBPropertiesSetter();
    private MessagePayloadProxy payloadProxy;

    public PackageJmsMessageContents(ConfigTree config) {
        this(createPayloadProxy(config)) ;
        String excludeProperties = config.getAttribute("excludeProperties");
        if (excludeProperties != null)
        {
            esbPropertiesStrategy = new DefaultESBPropertiesSetter(excludeProperties);
        }
    }
    
    public PackageJmsMessageContents(MessagePayloadProxy payloadProxy) {
        this.payloadProxy = payloadProxy;
    }

    /**
	 * Will just drop the jms message contents into a esb Message
	 * 
	 * @param obj An instance of Message	
	 * @return <code>esbMessage</code> A newly created ESB Message object populated with the contents 
	 * 									obj argument passed in.
	 * @throws JMSException
	 * @throws IOException
	 * @see #setESBMessageBody(javax.jms.Message, Message)
	 */
	public Message process (final Object obj) throws JMSException, IOException, MessageDeliverException {
		if ( !(obj instanceof javax.jms.Message))
			throw new IllegalArgumentException("Object must be instance of javax.jms.Message");
		
		final javax.jms.Message jmsMessage = (javax.jms.Message) obj;
		
		final Message esbMessage = MessageFactory.getInstance().getMessage();
		
		setESBMessageBody( jmsMessage, esbMessage );

        try {
            if ( payloadProxy.getPayload(esbMessage) == null ) {
                return null;
            }
        } catch (MessageDeliverException e) {
            return null;
        }

        setPropertiesFromJMSMessage ( jmsMessage, esbMessage );
		
		return esbMessage;
	}
	
	/**
	 * Set the {@link ESBPropertiesSetter} to be used
	 * 
	 * @param esbPropertiesStrategy the strategy to be used
	 */
	public void setEsbPropertiesStrategy( ESBPropertiesSetter esbPropertiesStrategy )
	{
		this.esbPropertiesStrategy = esbPropertiesStrategy;
	}	
	
	/**
	 * Delegates to {@link DefaultESBPropertiesSetter#setPropertiesFromJMSMessage(javax.jms.Message, Message)}
	 * by default, but this method can be overridden by subclasses that need a different behaviour. 
	 * </p>
	 * It is also possible to set a different strategy by setting {@link #setEsbPropertiesStrategy(ESBPropertiesSetter)}
	 */
	protected void setPropertiesFromJMSMessage( javax.jms.Message fromJMS, org.jboss.soa.esb.message.Message toESB ) throws JMSException
	{
		esbPropertiesStrategy.setPropertiesFromJMSMessage( fromJMS, toESB );
	}

	/**
	 * This method will set the ESB Message's body to the appropriate type matching
	 * the JMS Message fromJMSMessage argument. 
	 * <p>
	 * TextMessage -> Body.add ( String ); 
	 * BytesMessage -> Body.add ( byte[] ); 
	 * ObjectMessage -> Body.add ( Object ); 
	 * </p>
	 * Note: this method will always set the bodies byte array
	 * to be backward compatible.
	 * 
	 * @param fromJMSMessage
	 * @param toESBMessage
	 * @throws JMSException 
	 * @throws IOException 
	 */
	private void setESBMessageBody( final javax.jms.Message fromJMSMessage, final Message toESBMessage ) throws JMSException, IOException, MessageDeliverException {
		byte[] bodyAsBytes = null;
		
		if (fromJMSMessage instanceof TextMessage)
		{
			final String text = ((TextMessage) fromJMSMessage).getText();
			payloadProxy.setPayload(toESBMessage, text);
		}
		else if (fromJMSMessage instanceof BytesMessage)
		{
			final BytesMessage jBytes = (BytesMessage) fromJMSMessage;
			ByteArrayOutputStream out = new ByteArrayOutputStream();
			byte[] ba = new byte[1000];
			int iQread;
			while (-1 != (iQread = jBytes.readBytes(ba)))
			{
				if (iQread > 0) out.write(ba, 0, iQread);
					out.close();
			}
			bodyAsBytes = out.toByteArray();
            payloadProxy.setPayload(toESBMessage, bodyAsBytes);
		}
		else if (fromJMSMessage instanceof ObjectMessage)
		{
			final Object object = ((ObjectMessage) fromJMSMessage).getObject();
            payloadProxy.setPayload(toESBMessage, object);
		}
		else
        {
            log.warn("Message type " + fromJMSMessage.getClass().getSimpleName() + " not supported - Message is ignored");
        }
	}

    public static MessagePayloadProxy createPayloadProxy(ConfigTree config) {
        return  new MessagePayloadProxy(config,
                new String[] {BytesBody.BYTES_LOCATION},
                new String[] {Body.DEFAULT_LOCATION, BytesBody.BYTES_LOCATION});
    }
}
