/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.gateway;

import groovy.lang.Binding;
import groovy.lang.GroovyShell;
import groovy.lang.Script;

import java.io.IOException;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.scripting.GroovyActionProcessor;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.lifecycle.AbstractThreadedManagedLifecycle;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.soa.esb.listeners.message.BasicMessageComposer;
import org.jboss.soa.esb.listeners.message.UncomposedMessageDeliveryAdapter;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.util.ClassUtil;

/**
 * Gateway listener that start the supplied Groovy script.
 * <p/>
 * The Groovy script effectively becomes the gateway listener.  The script path (classpath)
 * is specified in the "script" configuration property.
 * <p/>
 * The script is injected with references to the
 * {@link org.w3c.dom.Document config} ("config"), a {@link org.jboss.soa.esb.listeners.message.UncomposedMessageDeliveryAdapter}
 * instance for its target service ("deliveryAdapter"), and a reference to the gateway lifecycle instance ("gateway").
 * <p/>
 * <h3>Sample Config</h3>
 * <pre>
 *  &lt;groovy-listener name="soap-client" script="/org/jboss/scripts/MyScript.groovy" is-gateway="true" /&gt;
 * </pre>
 * <h3>Sample Script</h3>
 * <pre>
 *  def stopped = false;
 *  println "Starting loop in listener..."
 *  while(!stopped) {
 *      print "."
 *      stopped = gateway.{@link GroovyGateway#waitUntilStopping(long) waitUntilStopping}(200);
 *  }
 *  println ""
 *  println "loop ended"
 *
 *  println "delivering message..."
 *  {@link UncomposedMessageDeliveryAdapter deliveryAdapter}.deliverAsync({@link org.w3c.dom.Document config}.getAttribute("xxxx"));
 * </pre>
 *
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class GroovyGateway extends AbstractThreadedManagedLifecycle {

    /**
     * Logger.
     */
    private static Logger logger = Logger.getLogger(GroovyGateway.class);
    /**
     * Gateway script.
     */
    private Script script;

    /**
     * Construct the threaded managed lifecycle.
     *
     * @param config The configuration associated with this instance.
     * @throws org.jboss.soa.esb.ConfigurationException
     *          for configuration errors during initialisation.
     */
    public GroovyGateway(ConfigTree config) throws ConfigurationException {
        super(config);
        String scriptPath = config.getAttribute("script");

        scriptPath = mapToPrebundledScript(scriptPath);
        if(scriptPath == null) {
            throw new ConfigurationException("No Groovy script specified on listener config " + config.getAttribute("name"));
        } else {
            try {
                GroovyShell shell = new GroovyShell();
                script = shell.parse(GroovyActionProcessor.getScriptFromClasspath(scriptPath));
            } catch (Exception e) {
                throw new ConfigurationException("Error loading Groovy script '" + scriptPath + "' stream.", e);
            }
        }
    }

    private String mapToPrebundledScript(String scriptName) {
        try {
            String scriptPath = ClassUtil.getPath(getClass().getPackage()) + "/scripts/" + scriptName;

            if(!scriptPath.endsWith(".groovy")) {
                scriptPath += ".groovy";
            }
            if(GroovyActionProcessor.getScriptFromClasspath(scriptPath) != null) {
                return scriptPath;
            }
        } catch (IOException e) {
            // Ignore...
        }

        return scriptName;
    }

    protected void doInitialise() throws ManagedLifecycleException {
        try {
            // Configure the groovy script....
            Binding binding = new Binding();
            UncomposedMessageDeliveryAdapter deliveryAdapter = createDeliveryAdapter();

            binding.setVariable("config", getConfig());
            binding.setVariable("gateway", this);
            binding.setVariable("listener", this);
            if(deliveryAdapter != null) {
                binding.setVariable("deliveryAdapter", deliveryAdapter);
            }
            script.setBinding(binding);
        } catch (ConfigurationException e) {
           throw new ManagedLifecycleException("Unable to construct message delivery adapter.", e);
        }
    }

    protected void doRun() {
        logger.info("Starting Groovy Gateway " + getConfig().getName());
        script.run();
        logger.info("Stopping Groovy Gateway " + getConfig().getName());
    }

    /**
     * Factory method for adapter creation.
     *
     * @return The adapter instance.
     * @throws ConfigurationException Configuration error.
     */
    protected UncomposedMessageDeliveryAdapter createDeliveryAdapter() throws ConfigurationException {
        if(getConfig().getAttribute(ListenerTagNames.TARGET_SERVICE_NAME_TAG) != null) {
            return UncomposedMessageDeliveryAdapter.getGatewayDeliveryAdapter(getConfig(), new BasicMessageComposer());
        } else {
            return null;
        }
    }
}
