/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.deployers.mc;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.management.ObjectName;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.publish.ContractReferencePublisher;
import org.jboss.soa.esb.listeners.config.ServiceContract;
import org.jboss.soa.esb.listeners.config.model.ModelAdapter;

/**
 * Metadata for an ESB deployment.
 * <p/>
 *
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 */
public class EsbMetaData implements Serializable
{
    /** Serial version unique identifier. */
    private static final long serialVersionUID = 0L;

    /** The name of the .esb archive. */
    private String archiveName;

    /** The ESB configuration xml. */
    private final String esbConfigXml;
    
    /** The deployments dependencies */
    private final Set<ObjectName> dependencies ;
    
    /** The deployment name. The .esb archive name without the .esb suffix */
    private final String deploymentName;
    
    /** The adapter for querying the model */
    private ModelAdapter model;
    
    /** The publishers for this deployment */
    private List<ContractReferencePublisher> publishers = new ArrayList<ContractReferencePublisher>();
    
    /** The service contracts for this deployment */
    private List<ServiceContract> serviceContracts = new ArrayList<ServiceContract>();
    
    /**
     * Sole constructor.
     *
     * @param esbConfigXml      The xml configuration for the ESB.
     * @param archiveName       The name of the .esb archive from which this deployment comes from.
     * @param dependencies      Dependencies of this deployment. This might be object names or simply bean names.
     * @param model             The ModelAdapter for the deployment.
     */
    public EsbMetaData(final String esbConfigXml, final String archiveName, final String deploymentName, final Set<ObjectName> dependencies, final ModelAdapter model)
    {
        AssertArgument.isNotNullAndNotEmpty(esbConfigXml, "esbConfigXml");
        AssertArgument.isNotNullAndNotEmpty(archiveName, "archiveName");
        AssertArgument.isNotNullAndNotEmpty(deploymentName, "deploymentName");
        
        this.esbConfigXml = esbConfigXml;
        this.archiveName = archiveName;
        this.deploymentName = deploymentName;
        this.dependencies = dependencies;
        this.model = model;
    }

    /**
     * Gets the name of the archive that this metadata came from.
     *
     * @return String   The name of the archive that this metadata came from.
     */
    public final String getArchiveName()
    {
        return archiveName;
    }

    /**
     * Gets the esb xml configuration. This is the actual xml
     * and not a file path or anything else.
     * @return String The esb xml configuration.
     */
    public String getEsbConfigXml()
    {
        return esbConfigXml;
    }
    
    /**
     * Gets a set of dependencies of this deployment.
     * 
     * @return Set<ObjectName> A set of object names that the deployment represented by this instance depends on.
     */
    public Set<ObjectName> getDependencies()
    {
        return Collections.unmodifiableSet(dependencies);
    }
    
    /**
     * The deployment name is the name of the .esb archive without the .esb suffix.
     * 
     * @return String The name of the deployment. This is the archive name without the .esb suffix.
     */
    public String getDeploymentName()
    {
        return deploymentName;
    }
    
    /**
     * The model adapter for the configuration.
     * 
     * @return ModelAdapter The model adapter for the configuration.
     */
    public ModelAdapter getModel()
    { 
        return model;
    }
    
    /**
     * Sets the publishers for this deployment.
     * 
     * @param publishers The publishers for this deployment.
     */
    public void setPublishers(final List<ContractReferencePublisher> publishers)
    {
        this.publishers = publishers;
    }
    
    /**
     * Gets the publishers for this deployment.
     * 
     * @return List<ContractReferencePublishers> List of publishers for this deployment.
     */
    public List<ContractReferencePublisher> getPublishers()
    {
        return Collections.unmodifiableList(publishers);
    }
    
    /**
     * Sets the service contracts for this deployment.
     * 
     * @param serviceContracts The service contracts for this deployment.
     */
    public void setServiceContracts(final List<ServiceContract> serviceContracts)
    {
        this.serviceContracts = serviceContracts;
    }
    
    /**
     * Gets the service contracts for this deployment.
     * 
     * @return List<ServiceContract> List of service contracts for this deployment.
     */
    public List<ServiceContract> getServiceContracts()
    {
        return Collections.unmodifiableList(serviceContracts);
    }

    /**
     * @return String   String representation of this metadata.
     */
    @Override
    public final String toString()
    {
        return String.format("EsbMetaData [archiveName='%s', deploymentName='%s', dependencies='%s']", archiveName, deploymentName, dependencies);
    }
}
