/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 */

package org.jboss.soa.esb.listeners.config.mappers110;

import java.io.File;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.eprs.FileEpr;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.FsBusDocument.FsBus;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.FsListenerDocument.FsListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.FsMessageFilterDocument.FsMessageFilter;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.FsProviderDocument.FsProvider;
import org.jboss.soa.esb.listeners.gateway.FileGatewayListener;
import org.w3c.dom.Element;

/**
 * Performs the mapping of a &lt;fs-listener&gt; XSD based configuration to the "ConfigTree"
 * style configuration, adding the "ConfigTree" listener config to the "root" node.
 * 
 * @author <a href="mailto:kurt.stam@jboss.com">kurt.stam@jboss.com</a>
 */
public class FsListenerMapper {

	/**
	 * Perform the mapping.
	 * @param root The "ConfigTree" configuration root node.
	 * @param listener The Jmslistener to be mapped into the ConfigTree.
	 * @param model The configuration model from which the mapping is being performed.
	 * @return The ConfigTree listener configuration node.
	 * @throws ConfigurationException Invalid listener configuration.
	 */
	public static Element map(Element root, FsListener listener, XMLBeansModel model) throws ConfigurationException {
		Element listenerNode = YADOMUtil.addElement(root, "listener");
		FsBus bus;
		FsProvider provider;

        listenerNode.setAttribute("name", listener.getName());

        try {
			bus = (FsBus) model.getBus(listener.getBusidref());
		} catch (ClassCastException e) {
			throw new ConfigurationException("Invalid busid reference [" + listener.getBusidref() + "] on listener [" + listener.getName() + "].  A <fs-listener> must reference a <fs-bus>.");
		}
		try {
			provider = (FsProvider) model.getProvider(bus);
		} catch (ClassCastException e) {
			throw new ConfigurationException("Invalid bus config [" + listener.getBusidref() + "].  Should be contained within a <fs-provider> instance.  Unexpected exception - this should have caused a validation error!");
		}
		
		FsMessageFilter messageFilter = listener.getFsMessageFilter();
		if(messageFilter == null) {
			messageFilter = bus.getFsMessageFilter();
			if(messageFilter == null) {
				throw new ConfigurationException("No <fs-detination> defined on either <fs-listener> [" + listener.getName() + "] or <fs-bus> [" + bus.getBusid() + "].");
			}
		}
		// Map the standard listener attributes - common across all listener types...
		MapperUtil.mapDefaultAttributes(listener, listenerNode, model);
		// Map the <property> elements targeted at the listener - from the listener itself.
		MapperUtil.mapProperties(listener.getPropertyList(), listenerNode);			
		if(listener.getIsGateway()) {
			listenerNode.setAttribute("gatewayClass", FileGatewayListener.class.getName());
			// Map EPR related attributes onto the listener - from the bus and provider and listener.
			// Note: This will change - the Gateways will also support the EPR element...
			mapFsEprProperties(listenerNode, provider, messageFilter);
			MapperUtil.mapEPRProperties(listener, listenerNode, model);
		} else {
			Element eprNode = YADOMUtil.addElement(listenerNode, ListenerTagNames.EPR_TAG);

			// Map EPR related attributes onto the EPR - from the bus and provider and listener...
			mapFsEprProperties(eprNode, provider, messageFilter);
			MapperUtil.mapEPRProperties(listener, eprNode, model);
			// Remove any empty attributes set on the EPR config...
			YADOMUtil.removeEmptyAttributes(eprNode);
		}
		
		// Remove any empty attributes set on the listener config...
		YADOMUtil.removeEmptyAttributes(listenerNode);
		
		return listenerNode;
	}

	private static void mapFsEprProperties(Element toElement, FsProvider provider, FsMessageFilter messageFilter) {
		final File directory = new File(messageFilter.getDirectory()) ;
		toElement.setAttribute(ListenerTagNames.URL_TAG, directory.toURI().toString());
		toElement.setAttribute(FileEpr.INPUT_SUFFIX_TAG, messageFilter.getInputSuffix());
		toElement.setAttribute(FileEpr.WORK_SUFFIX_TAG, messageFilter.getWorkSuffix());
		toElement.setAttribute(FileEpr.POST_DEL_TAG, String.valueOf(messageFilter.getPostDelete()));
		toElement.setAttribute(FileEpr.POST_RENAME_TAG, String.valueOf(messageFilter.getPostRename()));
		if (!messageFilter.getPostDelete()) {
			toElement.setAttribute(FileEpr.POST_DIR_TAG, messageFilter.getPostDirectory());
			toElement.setAttribute(FileEpr.POST_SUFFIX_TAG, messageFilter.getPostSuffix());
		}
		toElement.setAttribute(FileEpr.ERROR_DEL_TAG, String.valueOf(messageFilter.getErrorDelete()));
		if (!messageFilter.getErrorDelete()) {
			toElement.setAttribute(FileEpr.ERROR_DIR_TAG, messageFilter.getErrorDirectory());
			toElement.setAttribute(FileEpr.ERROR_SUFFIX_TAG, messageFilter.getErrorSuffix());
		}
	}
}
