/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2009
 */
package org.jboss.soa.esb.http;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Set;

import javax.management.AttributeNotFoundException;
import javax.management.JMException;
import javax.management.MBeanServer;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.jboss.mx.util.MBeanServerLocator;
import org.jboss.soa.esb.common.Configuration;
import org.jboss.soa.esb.common.Environment;

/**
 * HttpConfiguration.
 * 
 * @author dward at jboss.org
 */
public final class HttpConfiguration
{
	
	private static final Logger logger = Logger.getLogger(HttpConfiguration.class);
	
	private static MBeanServer mbeanServer = null;
	
	private HttpConfiguration() {}
	
	private static MBeanServer getMBeanServer()
	{
		if (mbeanServer == null)
		{
			synchronized(HttpConfiguration.class)
			{
				if (mbeanServer == null)
				{
					mbeanServer = MBeanServerLocator.locateJBoss();
				}
			}
		}
		return mbeanServer;
	}
	
	public static String getHost()
	{
		return getHost(Environment.DEFAULT_HOST);
	}
	
	public static String getHost(String defaultValue)
	{
		String global = Configuration.getHttpHost();
		if (global != null)
		{
			return global;
		}
		String bindAddress = Configuration.getBindAddress();
		if (bindAddress != null)
		{
			return bindAddress;
		}
		return defaultValue;
	}
	
	public static int getPort()
	{
		return getPort(8080);
	}
	
	public static int getPort(int defaultValue)
	{
		String global = Configuration.getHttpPort();
		if (global != null)
		{
			return Integer.valueOf(global).intValue();
		}
		int port = getConnectorPort("HTTP/1.1", false);
		if (port > -1)
		{
			return port;
		}
		return defaultValue;
	}
	
	public static int getSecurePort()
	{
		return getSecurePort(8443);
	}
	
	public static int getSecurePort(int defaultValue)
	{
		String global = Configuration.getHttpSecurePort();
		if (global != null)
		{
			return Integer.valueOf(global).intValue();
		}
		int port = getConnectorPort("HTTP/1.1", true);
		if (port > -1)
		{
			return port;
		}
		return defaultValue;
	}
	
	private static int getConnectorPort(String protocol, boolean secure)
	{
		int port = -1;
		try
		{
			MBeanServer mbs = getMBeanServer();
			Set<?> connectorNames = mbs.queryNames(new ObjectName("jboss.web:type=Connector,*"), null);
			for (Object current : connectorNames)
			{
				ObjectName currentName = (ObjectName)current;
				try
				{
					int connectorPort = (Integer)mbs.getAttribute(currentName, "port");
					boolean connectorSecure = (Boolean)mbs.getAttribute(currentName, "secure");
					String connectorProtocol = (String)mbs.getAttribute(currentName, "protocol");
					if (protocol.equals(connectorProtocol) && secure == connectorSecure)
					{
						if (port > -1)
						{
							logger.warn("Found multiple connectors for protocol='" + protocol + "' and secure='" + secure + "', using first port found '" + port + "'");
						}
						else
						{
							port = connectorPort;
						}
					}
				}
				catch (AttributeNotFoundException ignored) {}
			}
		}
		catch (JMException ignored)
		{
			port = -1;
		}
		return port;
	}
	
	public static String getAddress()
	{
		return getAddress(null);
	}
	
	public static String getAddress(String path)
	{
		return buildAddress(path, false);
	}
	
	public static String getSecureAddress()
	{
		return getSecureAddress(null);
	}
	
	public static String getSecureAddress(String path)
	{
		return buildAddress(path, true);
	}
	
	private static String buildAddress(String path, boolean secure)
	{
		StringBuilder sb = new StringBuilder();
		if (secure)
		{
			sb.append( "https://" );
			sb.append( getHost() );
			sb.append( ":" );
			sb.append( getSecurePort() );
		}
		else
		{
			sb.append( "http://" );
			sb.append( getHost() );
			sb.append( ":" );
			sb.append( getPort() );
		}
		if (path != null)
		{
			path = path.trim();
			if (path.length() > 0)
			{
				if ( !path.startsWith("/") )
				{
					sb.append("/");
				}
				sb.append(path);
			}
		}
		return sb.toString();
	}
	
	public static URL getURL() throws MalformedURLException
	{
		return new URL( getAddress() );
	}
	
	public static URL getURL(String path) throws MalformedURLException
	{
		return new URL( getAddress(path) );
	}
	
	public static URL getSecureURL() throws MalformedURLException
	{
		return new URL( getSecureAddress() );
	}
	
	public static URL getSecureURL(String path) throws MalformedURLException
	{
		return new URL( getSecureAddress(path) );
	}
	
	public static URI getURI() throws URISyntaxException
	{
		return new URI( getAddress() );
	}
	
	public static URI getURI(String path) throws URISyntaxException
	{
		return new URI( getAddress(path) );
	}
	
	public static URI getSecureURI() throws URISyntaxException
	{
		return new URI( getSecureAddress() );
	}
	
	public static URI getSecureURI(String path) throws URISyntaxException
	{
		return new URI( getSecureAddress(path) );
	}
	
	public static URI createURI()
	{
		return URI.create( getAddress() );
	}
	
	public static URI createURI(String path)
	{
		return URI.create( getAddress(path) );
	}
	
	public static URI createSecureURI()
	{
		return URI.create( getSecureAddress() );
	}
	
	public static URI createSecureURI(String path)
	{
		return URI.create( getSecureAddress(path) );
	}
	
	public static boolean isResponseStatusEnabled()
	{
		return isResponseStatusEnabled(true);
	}
	
	public static boolean isResponseStatusEnabled(boolean defaultValue)
	{
		String global = Configuration.getHttpResponseStatusEnabled();
		if (global != null)
		{
			return Boolean.valueOf(global).booleanValue();
		}
		return defaultValue;
	}

}
