/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.helpers.persist;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.log4j.Logger;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.Configuration;
import org.hibernate.cfg.Environment;
import org.jboss.soa.esb.ConfigurationException;

/**
 * HibernateSessionFactory is a singleton for hibernate's SessionFactory object.    The Listener
 * uses JNDI to store the Hibernate SessionFactory.
 * 
 * @author <a href="mailto:tcunning@redhat.com">tcunning@redhat.com</a>
 * @since Version 4.2
 * 
 */
public class HibernateSessionFactory {
	public static final String HIBERNATE_JNDI = "java:comp/env/hibernate/SessionFactory";
	
	private static final Logger m_Logger = Logger.getLogger(HibernateSessionFactory.class);	
	
	private HibernateSessionFactory() {
	}
				
	public static SessionFactory getInstance(Configuration f_cfg) throws ConfigurationException {
		return init(f_cfg);		
	}
		
	/**
	 * Checks whether the SessionFactory exists within JNDI and whether it is closed.
	 * @param f_cfg Hibernate Configuration - needed for SESSION_FACTORY_NAME
	 * @return whether the Hibernate SessionFactory is in JNDI and is alive
	 */
	public static boolean isAlive(Configuration f_cfg) {
		boolean result = false; 
		SessionFactory sf = null;
		String cfgName = f_cfg.getProperty(Environment.SESSION_FACTORY_NAME);

		Context ic = null;
		// Look up SessionFactory in JNDI
		if (cfgName != null) {
			try {
				ic = new InitialContext();
				sf = (SessionFactory) ic.lookup(cfgName);
				result = ! sf.isClosed();
				m_Logger.debug("Connection isAlive is " + result + ".");
			} catch (NamingException ne) {
			}
		}
		return result;
	}
	
	/**
	 * Grab InitialContext out of JNDI.
	 * @param f_cfg hibernate configuration
	 * @throws ConfigurationException
	 */
	private static SessionFactory init(Configuration f_cfg) throws ConfigurationException {
		String cfgName = f_cfg.getProperty(Environment.SESSION_FACTORY_NAME);
		SessionFactory sf = null;
		Context ic = null;
		// Look up SessionFactory in JNDI
		if (cfgName != null) {
			try {
				ic = new InitialContext();
				sf = (SessionFactory) ic.lookup(cfgName);
				m_Logger.debug("Found SessionFactory in JNDI.");
			} catch (NamingException ne) {
				m_Logger.debug("Could not find SessionFactory in JNDI.", ne);
			}
		} else {
			f_cfg.setProperty(Environment.SESSION_FACTORY_NAME, HIBERNATE_JNDI);
			cfgName = HIBERNATE_JNDI;
		}
		
		if (sf == null) {
            // Let Hibernate bind it to JNDI
			m_Logger.debug("Build SessionFactory from Configuration.");
            sf = f_cfg.buildSessionFactory();
		}
		return sf;
	}

	/**
	 * Close the SessionFactory stored in JNDI at SESSION_FACTORY_NAME.
	 * @param f_cfg hibernate configuration
	 */
	public static void close(Configuration f_cfg) {
		SessionFactory sf = null;
		
		String cfgName = f_cfg.getProperty(Environment.SESSION_FACTORY_NAME);

		Context ic = null;
		// Look up SessionFactory in JNDI
		if (cfgName != null) {
			try {
				ic = new InitialContext();
				sf = (SessionFactory) ic.lookup(cfgName);
			} catch (Exception e) {
			}
		}
		
		if (sf != null) {
			sf.close();	
		}
		sf = null;
	}
}