/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */
package org.jboss.soa.esb.common;

import javax.transaction.Status;
import javax.transaction.Synchronization;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;
import javax.transaction.xa.XAResource;

import org.apache.log4j.Logger;
import org.jboss.system.ServiceMBeanSupport;
import org.jboss.tm.TransactionManagerLocator;

public class JBossESBTransactionService extends ServiceMBeanSupport implements JBossESBTransactionServiceMBean
{
    @Override
    protected void createService()
        throws Exception
    {
        TransactionStrategy.setTransactionStrategy(new JTATransactionStrategy()) ;
    }
    
    /**
     * The JTA transaction strategy used in the application server.
     * @author kevin
     */
    private static class JTATransactionStrategy extends TransactionStrategy
    {
        /**
         * The logger for this class.
         */
        private static final Logger LOGGER = Logger.getLogger(JTATransactionStrategy.class) ;
        
        /**
         * The transaction manager.
         */
        private final TransactionManager tm ;
        
        /**
         * Construct the JTA transaction strategy.
         */
        JTATransactionStrategy()
        {
            tm = TransactionManagerLocator.getInstance().locate() ;
        }
        
        /**
         * Begin a transaction on the current thread.
         * @throws TransactionStrategyException
         */
        public void begin()
            throws TransactionStrategyException
        {
            try
            {
                tm.begin() ;
                LOGGER.trace("Transaction started on current thread") ;
            }
            catch (final Throwable th)
            {
                LOGGER.debug("Failed to start transaction on current thread", th) ;
                throw new TransactionStrategyException("Failed to begin transaction on current thread", th) ;
            }
        }
        
        /**
         * Terminate the transaction on the current thread.
         * If the transaction has been marked for rollback then it
         * will be rolled back, otherwise it will be committed.
         * @throws TransactionStrategyException
         */
        public void terminate()
            throws TransactionStrategyException
        {
            try
            {
                final int status = tm.getStatus() ;
                switch(status)
                {
                    case Status.STATUS_ACTIVE:
                        LOGGER.trace("Committing transaction on current thread") ;
                        tm.commit() ;
                        break ;
                    case Status.STATUS_MARKED_ROLLBACK:
                        LOGGER.trace("Rollback transaction on current thread") ;
                        tm.rollback();
                        break ;
                    default:
                        // Transaction is in some other state, just disassociate
                        if (LOGGER.isTraceEnabled())
                        {
                            LOGGER.trace("Suspending transaction on current thread, status: " + status) ;
                        }
                        tm.suspend() ;
                }
            }
            catch (final Throwable th)
            {
                LOGGER.debug("Failed to terminate transaction on current thread", th) ;
                throw new TransactionStrategyException("Failed to terminate transaction on current thread", th) ;
            }
        }
        
        /**
         * Mark the current transaction for rollback.
         * @throws TransactionStrategyException
         */
        public void rollbackOnly()
            throws TransactionStrategyException
        {
            try
            {
                tm.setRollbackOnly();
            }
            catch (final Throwable th)
            {
                throw new TransactionStrategyException("Failed to mark the transaction on current thread for rollback", th) ;
            }
        }
        
        /**
         * Get a handle on the currently associated transaction (or null).
         * @throws TransactionStrategyException
         */
        
        public Object getTransaction () throws TransactionStrategyException
        {
        	try
        	{
        		return tm.getTransaction();
        	}
        	catch (final Throwable th)
        	{
        		throw new TransactionStrategyException("Problem when trying to getTransaction: ",th);
        	}
        }
        
        /**
         * Suspend the current thread-to-transaction association.
         * 
         * @return the associated transaction, or null.
         * @throws TransactionStrategyException
         */
        public Object suspend () throws TransactionStrategyException
        {
        	try
        	{
        		return tm.suspend();
        	}
        	catch (final Throwable th)
        	{
        		throw new TransactionStrategyException("Problem when trying to suspend transaction: ",th);
        	}
        }
        
        /**
         * Associated the transaction with the current thread.
         * @param tx
         * @throws TransactionStrategyException
         */
        public void resume (Object tx) throws TransactionStrategyException
        {
        	try
        	{
        		tm.resume((Transaction) tx);
        	}
        	catch (final Throwable th)
        	{
        		throw new TransactionStrategyException("Problem when trying to resume transaction: ",th);
        	}
        }
        
        /**
         * Add a synchronization to the current transaction.
         * @param sync
         * @throws TransactionStrategyException
         */
        public void registerSynchronization (Synchronization sync) throws TransactionStrategyException
        {
            try
            {
        	tm.getTransaction().registerSynchronization(sync);
            }
            catch (final Throwable th)
            {
        	throw new TransactionStrategyException("Problem when registering synchronization: ", th);
            }
        }
        
        /**
         * Add a resource to the current transaction.
         * @param resource
         * @throws TransactionStrategyException
         */
        public void enlistResource(XAResource resource) throws TransactionStrategyException
        {
            try
            {
                tm.getTransaction().enlistResource(resource);
            }
            catch (final Throwable th)
            {
                throw new TransactionStrategyException("Problem when enlisting resource", th);
            }
        }
        
        /**
         * Is the currently associated transaction active?
         * @return
         * @throws TransactionStrategyException
         */
        public boolean isActive () throws TransactionStrategyException
        {
        	try
        	{
        		if (tm.getStatus() == Status.STATUS_ACTIVE)
        			return true;
        		else
        			return false;
        	}
        	catch (final Throwable th)
        	{
        		throw new TransactionStrategyException("Problem when trying to get transaction status: ",th);
        	}
        }
        
        public String toString ()
        {
        	return "JTATransactionStrategy";
        }
    }

}
