/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.addressing.eprs;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.Service;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.PortReference;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.w3c.dom.Element;

import java.net.URI;
import java.net.URISyntaxException;

/**
 * Logical EPR.
 * <p/>
 * Logical EPRs allow services to be addressed by their "logical" (Vs physical)
 * name.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class LogicalEPR extends EPR {

    private static final String PROTOCOL = "logical";
    private ServiceInvoker serviceInvoker;

    public LogicalEPR() {
        super();
    }

    public LogicalEPR(PortReference addr) {
    	this(new EPR(addr)) ;
    }

    public LogicalEPR(URI uri) {
    	this(new EPR(uri)) ;
    }

    public LogicalEPR(final EPR epr) {
        super(epr);
        assertValidLogicalURI(URI.create(((PortReference)AssertArgument.isNotNull(epr.getAddr(), "addr")).getAddress()));
    }

    public LogicalEPR(final EPR epr, final Element header) {
        this(epr);
    }

    public LogicalEPR(String serviceCategory, String serviceName) {
        super(toLogicalURI(serviceCategory, serviceName));
    }

    public void setAddr(PortReference addr) {
        AssertArgument.isNotNull(addr, "addr");
        assertValidLogicalURI(URI.create(addr.getAddress()));
        super.setAddr(addr);
    }

    public Service toService() {
        PortReference portRef = getAddr();

        if(portRef == null) {
            throw new IllegalStateException("Invalid Logical EPR state.  No PortReference set on EPR.");
        }

        URI uri;
        try {
            uri = new URI(portRef.getAddress());
        } catch (URISyntaxException e) {
            throw new IllegalStateException("Invalid Logical EPR state.  '" + portRef.getAddress() + "' is not a valid URI.", e);
        }

        assertValidLogicalURI(uri);

        return new Service(uri.getSchemeSpecificPart(), uri.getFragment());
    }

    public ServiceInvoker getServiceInvoker() throws MessageDeliverException {
        if(serviceInvoker == null) {
            serviceInvoker = new ServiceInvoker(toService(), getAddr().getAllExtensions());
        }
        return serviceInvoker;
    }

    private static URI toLogicalURI(String serviceCategory, String serviceName) {
        AssertArgument.isNotNullAndNotEmpty(serviceCategory, "serviceCategory");
        AssertArgument.isNotNullAndNotEmpty(serviceName, "serviceName");

        try {
            // Using this URI constructor because it encodes the
            // scheme specific part and fragment - the cat and name
            // may have illegal URI characters...
            return new URI(PROTOCOL, serviceCategory, serviceName);
        } catch (URISyntaxException e) {
            throw new IllegalArgumentException(e);
        }
    }

    private void assertValidLogicalURI(URI uri) {
        if(!PROTOCOL.equalsIgnoreCase(uri.getScheme())) {
            throw new IllegalArgumentException("'" + uri + "' is not a valid URI for a Logical EPR - URI scheme must be '" + PROTOCOL + "'.");
        }
        if(!uri.isOpaque()) {
            throw new IllegalArgumentException("'" + uri + "' is not a valid URI for a Logical EPR - URI must be opaque.");
        }

        String category = uri.getSchemeSpecificPart();
        String name = uri.getFragment();

        if(category == null) {
            throw new IllegalArgumentException("'" + uri + "' is not a valid URI for a Logical EPR - no URI scheme specific part (service category).");
        }
        if(name == null) {
            throw new IllegalArgumentException("'" + uri + "' is not a valid URI for a Logical EPR - no URI fragment (service name) part.");
        }
    }

	public EPR copy ()
	{
	    return new LogicalEPR(this);
	}
	
    public static URI type() {
        return _type;
    }

    private static URI _type;

    static {
        try {
            _type = new URI("urn:jboss/esb/epr/type/logical");
        }
        catch (Exception ex) {
            ex.printStackTrace();

            throw new ExceptionInInitializerError(ex.toString());
        }
    }

}
