package org.jboss.soa.esb.addressing.eprs;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */


/**
 * This class represents the endpoint reference for services.
 */


import java.net.URI;
import java.net.URISyntaxException;

import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.PortReference;
import org.jboss.soa.esb.addressing.XMLUtil;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * A helper class for using email style EPRs. Simply create and use instances of
 * this type.
 * 
 * @author marklittle
 *
 */
public class EmailEpr extends EPR
{  
	public static final String DEFAULT_PORT = "25";
	public static final String DEFAULT_USERNAME = "";
	public static final String DEFAULT_PASSWORD = "";
	
	public static final String SMTP_PROTOCOL = "smtp";
	public static final String POP_PROTOCOL = "pop";
	
	public static final String USERNAME_TAG = "username";
	public static final String PASSWORD_TAG = "password";
	
	private static final String PROTOCOL_SEPARATOR = "://";
	private static final String PORT_SEPARATOR = ":";
	
	public EmailEpr (EPR epr)
	{
		super(epr);
	}
	
	public EmailEpr (EPR epr, Element header)
	{
		super(epr);
		
		NodeList nl = header.getChildNodes();

		for (int i = 0; i < nl.getLength(); i++)
		{
			try
			{
				String prefix = nl.item(i).getPrefix();
				String tag = nl.item(i).getLocalName();
				
				if ((prefix != null) && (prefix.equals(XMLUtil.JBOSSESB_PREFIX)))
				{
					if ((tag != null) && (tag.equals(USERNAME_TAG)))
					{
						getAddr().addExtension(USERNAME_TAG, nl.item(i).getTextContent());
					}
					else
					{
						if ((tag != null) && (tag.equals(PASSWORD_TAG)))
							getAddr().addExtension(PASSWORD_TAG, nl.item(i).getTextContent());
					}
				}
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}
		}
	}
	
	/**
	 * Create a new email EPR. The port number will be assumed to be 25,
	 * and there are no values for username and password.
	 * 
	 * @param protocol the protocol to use.
	 * @param host the host name.
	 */
	
	public EmailEpr (String protocol, String host)
	{
		this(protocol, host, DEFAULT_PORT, DEFAULT_USERNAME, DEFAULT_PASSWORD);
	}
	
	/**
	 * Create a new email EPR.
	 * 
	 * @param protocol the protocol to use.
	 * @param host the host name.
	 * @param port the port to use.
	 * @param username the username for sending/receiving.
	 * @param password the password for sending/receiving.
	 */
	
	public EmailEpr (String protocol, String host, String port, String username, String password)
	{
		// how many of these do we really need? modify accordingly.
		
		if ((protocol == null) || (host == null) || (port == null))
			throw new IllegalArgumentException();
		
		if ((protocol.equals(SMTP_PROTOCOL) || (protocol.equals(POP_PROTOCOL))))
		{
			PortReference addr = new PortReference(protocol+PROTOCOL_SEPARATOR+host+PORT_SEPARATOR+port);
			
			if (username != null)
				addr.addExtension(USERNAME_TAG, username);
			
			if (password != null)
				addr.addExtension(PASSWORD_TAG, password);
			
			setAddr(addr);
		}
		else
			throw new IllegalArgumentException("Invalid email protocol!");
	}
	
	/**
	 * @return the email protocol used.
	 */
	
	public final String getProtocol ()
	{
            try
            {
		URI addr = new URI(getAddr().getAddress());
		
		return addr.getScheme();
            }
            catch (URISyntaxException ex)
            {
                _logger.warn("Unexpected parsing exception!", ex);
                
                return null;
            }
	}
	
	/**
	 * @return the email host used.
	 */
	
	public final String getHost ()
	{
            try
            {
		URI addr = new URI(getAddr().getAddress());
		
		return addr.getHost();
            }
            catch (URISyntaxException ex)
            {
                _logger.warn("Unexpected parsing exception!", ex);
                
                return null;
            }
	}
	
	/**
	 * @return the email port used, or -1 if not specified.
	 */

	public final int getPort ()
	{
            try
            {
		URI addr = new URI(getAddr().getAddress());
		
		return addr.getPort();
            }
            catch (URISyntaxException ex)
            {
                _logger.warn("Unexpected parsing exception!", ex);
                
                return -1;
            }
	}
	
	/*
	 * There are deliberately no setters for the values once the EPR is created.
	 */
	
	/**
	 * @return the password for this EPR, or <code>null</code> if none is set.
	 */
	
	public final String getPassword ()
	{
		return getAddr().getExtensionValue(PASSWORD_TAG);
	}
	
	/**
	 * @return the username for this EPR, or <code>null</code> if none is set.
	 */
	
	public final String getUserName ()
	{
		return getAddr().getExtensionValue(USERNAME_TAG);
	}

	public String toString ()
	{
		return "EmailEpr [ "+super.getAddr().extendedToString()+" ]";
	}

	public EPR copy ()
	{
	    return new EmailEpr(this);
	}
	
	public static final URI type ()
	{
	    return _type;
	}
	
	private static URI _type;

	static
	{
	    try
		{
		    _type = new URI("urn:jboss/esb/epr/type/email");
		}
		catch (Exception ex)
		{
		    ex.printStackTrace();
		    
		    throw new ExceptionInInitializerError(ex.toString());
		}
	}
} 