/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.actions.routing;

import javax.jms.JMSException;
import javax.naming.NamingException;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.actions.AbstractActionPipelineProcessor;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.actions.ActionUtils;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.body.content.BytesBody;

/**
 * Abstract router.
 * <p/>
 * Handles unwrapping of the message payload (if required) before routing.  To turn on
 * unwrapping, supply an "unwrap" property with a value of "true" (default "false").
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public abstract class AbstractRouter extends AbstractActionPipelineProcessor {

	/**
	 * Unwrap the message payload before routing. 
	 */
	public boolean unwrap = false;
    protected MessagePayloadProxy payloadProxy;

    /**
     * Public constructor.
     * @param config Action config.
     * @throws ConfigurationException Queue name not configured.
     * @throws JMSException Unable to configure JMS destination.
     * @throws NamingException Unable to configure JMS destination.
     */
    public AbstractRouter(ConfigTree config) throws ConfigurationException {
    	unwrap = config.getAttribute("unwrap", "false").equals("true");
        payloadProxy = new MessagePayloadProxy(config,
                                               new String[] {BytesBody.BYTES_LOCATION, ActionUtils.POST_ACTION_DATA},
                                               new String[] {ActionUtils.POST_ACTION_DATA});
    }

    /* (non-Javadoc)
     * @see org.jboss.soa.esb.actions.ActionProcessor#process(java.lang.Object)
     */
    public Message process(Message message) throws ActionProcessingException {
    	if(unwrap) {
            Object content;
            
            try {
                content = payloadProxy.getPayload(message);
            } catch (MessageDeliverException e) {
                throw new ActionProcessingException(e);
            }

            route(content);
    	} else {
    		route(message);
    	}
    	
    	return null;
    }

    public MessagePayloadProxy getPayloadProxy() {
        return payloadProxy;
    }

    /**
     * Route the message or message payload.
     * <p/>
     * If the message is to be unwrapped, it will have been unwrapped at this stage.
     * @param object The object to route.
     * @throws ActionProcessingException Exception during routing operation.
     */
    public abstract void route(Object object) throws ActionProcessingException;
}
