/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.actions.naming.strategy;

import groovy.lang.Binding;
import groovy.lang.GroovyShell;

import java.io.IOException;

import org.jboss.soa.esb.actions.scripting.GroovyActionProcessor;
import org.jboss.soa.esb.message.Message;
import org.w3c.dom.Node;

/**
 * Naming strategy that executes a Groovy script to manipulate a filename.
 * 
 * @author Derek Adams
 */
public class GroovyNamingStrategy extends AbstractFileNamingStrategy {

	/** Path to the groovy script that will be executed */
	private String scriptPath = DEFAULT_SCRIPT_PATH;

	/** Default path for the groovy script */
	public static final String DEFAULT_SCRIPT_PATH = "naming.groovy";

	/** XML attribute that holds the script pathx */
	public static final String ATTR_SCRIPT_PATH = "scriptPath";

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jboss.soa.esb.actions.naming.strategy.FileNamingStrategy#configure(org.w3c.dom.Node)
	 */
	public void configure(Node xml) {
		Node scriptPathNode = xml.getAttributes().getNamedItem(ATTR_SCRIPT_PATH);
		if (scriptPathNode != null) {
			setScriptPath(scriptPathNode.getNodeValue());
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jboss.soa.esb.actions.naming.strategy.AbstractFileNamingStrategy#process(java.lang.String,
	 *      org.jboss.soa.esb.message.Message)
	 */
	@Override
	public String process(String filename, Message message) {
		try {
			// Read the script into a string.
			String script = GroovyActionProcessor.getScriptFromClasspath(getScriptPath());

			// Bind objects into the Groovy context.
			Binding binding = new Binding();
			binding.setVariable("filename", filename);
			binding.setVariable("message", message);

			// Create a shell and evaluate
			GroovyShell shell = new GroovyShell(Thread.currentThread().getContextClassLoader(),
					binding);
			Object result = shell.evaluate(script);
			if (!(result instanceof String)) {
				throw new RuntimeException(
						"Groovy script should return a String indicating processed filename");
			}
			return (String) result;
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Set the path to the script.
	 * 
	 * @param scriptPath
	 */
	public void setScriptPath(String scriptPath) {
		this.scriptPath = scriptPath;
	}

	/**
	 * Get the path to the script.
	 * 
	 * @return String
	 */
	public String getScriptPath() {
		return scriptPath;
	}
}