/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.actions.converters;

import java.io.UnsupportedEncodingException;
import java.util.List;

import org.jboss.soa.esb.actions.AbstractActionPipelineProcessor;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.actions.ActionUtils;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;

/**
 * Byte Array to String processor.
 * <p/>
 * Sample Action Configuration:
 * <pre>
 * &lt;Action name="Bytes-To-UTF-8-String" processor="ByteArrayToString"&gt;
 *     &lt;property name="encoding" value="UTF-8" /&gt; &lt;!-- Default of "UTF-8". --&gt;
 * &lt;/Action&gt;
 * </pre>
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 * @since Version 4.0
 */
public class ByteArrayToString extends AbstractActionPipelineProcessor {
    
    private String encoding;
    private MessagePayloadProxy payloadProxy;

    /**
     * Public constructor.
     * @param config Action properties.
     */
    public ByteArrayToString(ConfigTree config) {
        encoding = config.getAttribute("encoding", "UTF-8");
        payloadProxy = new MessagePayloadProxy(config,
                                               new String[] {BytesBody.BYTES_LOCATION, ActionUtils.POST_ACTION_DATA},
                                               new String[] {ActionUtils.POST_ACTION_DATA});
    }

    /* (non-Javadoc)
     * @see org.jboss.soa.esb.actions.ActionProcessor#process(java.lang.Object)
     */
    public Message process(Message message) throws ActionProcessingException {
        byte[] bytes;
        
        try {
            bytes = (byte[]) payloadProxy.getPayload(message);
        } catch(ClassCastException e) {
            throw new ActionProcessingException("Message must be an array of bytes. Is " + message.getClass().getName());
        } catch (MessageDeliverException e) {
            throw new ActionProcessingException(e);
        }

        try {
            payloadProxy.setPayload(message, new String(bytes, encoding));
        	return message;
        } catch (UnsupportedEncodingException e) {
            throw new ActionProcessingException("Unable to decode byte[] to String. Unsupported character encoding configuration: " + encoding, e);
        } catch (MessageDeliverException e) {
            throw new ActionProcessingException(e);
        }
    }
}
