/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.actions.cbr;

import org.jboss.soa.esb.services.routing.cbr.ContentBasedRouter;
import org.jboss.soa.esb.services.routing.MessageRouterException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.mapping.ObjectMapper;
import org.jboss.soa.esb.message.mapping.ObjectMappingException;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.actions.ContentBasedWiretap;
import org.jboss.soa.esb.ConfigurationException;
import org.apache.log4j.Logger;

import java.util.*;
import java.io.*;

/**
 * Abstract Property Rules Router.
 * <p/>
 * Abstract base class for rules based routers that define their rules in
 * in {@link Properties} based formats (.properties files).
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
abstract class AbstractPropertyRulesRouter implements ContentBasedRouter {

    protected Logger logger;
    private String ruleSetConfig;
    private File ruleSetFile;
    private long lastLoaded = 0L;
    private Map<String, RoutingRule> routingMap;
    private MessagePayloadProxy payloadProxy;

    protected AbstractPropertyRulesRouter() {
        logger = Logger.getLogger(getClass());
    }

    public void setConfigTree(ConfigTree configTree) throws MessageRouterException {
        ruleSetConfig = configTree.getAttribute(ListenerTagNames.RULE_SET_TAG);
        if(ruleSetConfig != null) {
            ruleSetFile = new File(ruleSetConfig);
            routingMap = buildRoutingMap(loadRules());
        } else {
            ConfigTree[] routeToConfigs = configTree.getChildren(ContentBasedWiretap.ROUTE_TO_TAG);

            Properties rules = new Properties();
            for(ConfigTree routeToConfig : routeToConfigs) {
                try {
                    String destinationName = ContentBasedWiretap.buildDestinationKey(routeToConfig);
                    String expression = routeToConfig.getRequiredAttribute("expression");

                    rules.setProperty(destinationName, expression);
                } catch (ConfigurationException e) {
                    throw new MessageRouterException("Error processing <rule> configuration.", e);
                }
            }
            routingMap = buildRoutingMap(rules);
        }

        payloadProxy = new MessagePayloadProxy(configTree);
    }

    public Map<String, RoutingRule> getRoutingMap() {
        return routingMap;
    }

    public abstract Map<String, RoutingRule> buildRoutingMap(Properties rules) throws MessageRouterException;

    public List<String> route(String ruleSet, String ruleLanguage, boolean ruleReload, Message message, List<Object> objectList) throws MessageRouterException {
        return route(ruleSet, ruleReload, message, objectList);
    }

    public List<String> route(String ruleSet, boolean ruleReload, Message message, List<Object> objectList) throws MessageRouterException {
        if(ruleReload && ruleSetFile != null) {
            if(ruleSetFile.exists() && ruleSetFile.lastModified() != lastLoaded) {
                routingMap = buildRoutingMap(loadRules());
            }
        }

        List<Object> objectsToTest = getMessageObjectsToTest(message, objectList);
        List<String> destinations = new ArrayList<String>();
        Set<Map.Entry<String, RoutingRule>> routingRules = routingMap.entrySet();

        for(Map.Entry<String, RoutingRule> routingRule : routingRules) {
            for(Object objectToTest : objectsToTest) {
                if(routingRule.getValue().evaluate(objectToTest)) {
                    destinations.add(routingRule.getKey());
                }
            }
        }

        return destinations;
    }

    private Properties loadRules() throws MessageRouterException {
        InputStream ruleStream;

        // Get the properties file stream...
        if(ruleSetFile.exists()) {
            try {
                ruleStream = new FileInputStream(ruleSetFile);
            } catch (FileNotFoundException e) {
                throw new MessageRouterException("Unable to open rule properties file '" + ruleSetFile.getAbsolutePath() + "'.", e);
            }
            lastLoaded = ruleSetFile.lastModified();
        } else {
            ruleStream = ClassUtil.getResourceAsStream(ruleSetConfig, AbstractPropertyRulesRouter.class);
        }

        if(ruleStream == null) {
            throw new MessageRouterException("Unable to open rule properties file '" + ruleSetConfig + "'.");
        }

        // Load the rule properties...
        Properties rules = new Properties();
        try {
            try {
                rules.load(ruleStream);
            } catch (IOException e) {
                throw new MessageRouterException("Error reading rule properties file '" + ruleSetConfig + "'.", e);
            }
        } finally {
            try {
                ruleStream.close();
            } catch (IOException e) {
                throw new MessageRouterException("Error closing rule properties file '" + ruleSetConfig + "'.", e);
            }
        }

        return rules;
    }

    private List<Object> getMessageObjectsToTest(Message message, List<Object> objectList) throws MessageRouterException {
        List<Object> objectsToTest = new ArrayList<Object>();

        if(objectList == null || objectList.isEmpty()) {
            try {
                objectsToTest.add(payloadProxy.getPayload(message));
            } catch (MessageDeliverException e) {
                throw new MessageRouterException("Failed to get message payload from message.", e);
            }
        } else {
            ObjectMapper objectMapper = new ObjectMapper();

            for(Object object : objectList) {
                if(object instanceof String) {
                    try {
                        Object mappedObject = objectMapper.getObjectFromMessage(message, (String) object);

                        if(mappedObject == null) {
                            logger.debug("Object '" + object + "' not present on message.");
                            continue;
                        }

                        objectsToTest.add(mappedObject);
                    } catch (ObjectMappingException e) {
                        logger.debug("Object '" + object + "' not present on message.");
                    }
                }
            }
        }

        return objectsToTest;
    }
}
