/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.actions;

import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.common.TransactionStrategy;
import org.jboss.soa.esb.common.TransactionStrategyException;
import org.jboss.soa.esb.addressing.Call;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.apache.log4j.Logger;

/**
 * Synchronous Service Invoker.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class SyncServiceInvoker extends AbstractActionPipelineProcessor {

    private static Logger logger = Logger.getLogger(SyncServiceInvoker.class);

    private ServiceInvoker serviceInvoker;
    private long timeout;
    private boolean failOnException;
    private boolean suspendTransaction;

    public SyncServiceInvoker(ConfigTree configTree) throws ConfigurationException, MessageDeliverException {
        String serviceCat = configTree.getRequiredAttribute(ListenerTagNames.SERVICE_CATEGORY_NAME_TAG);
        String serviceName = configTree.getRequiredAttribute(ListenerTagNames.SERVICE_NAME_TAG);

        serviceInvoker = new ServiceInvoker(serviceCat, serviceName);
        timeout = configTree.getLongAttribute(ListenerTagNames.SERVICE_INVOKER_TIMEOUT, 30000L);
        failOnException = configTree.getBooleanAttribute(ListenerTagNames.FAIL_ON_EXCEPTION, true);
        suspendTransaction = configTree.getBooleanAttribute(ListenerTagNames.SUSPEND_TRANSACTION, false);
    }

    public Message process(Message message) throws ActionProcessingException {
        // Zap the faulto and replyto EPRs before sending...
        Call call = message.getHeader().getCall();
        EPR faultTo = call.getFaultTo();
        EPR replyTo = call.getReplyTo();
        Object txObject = null;

        if(isTransactional() && suspendTransaction) {
            try {
                txObject = TransactionStrategy.getTransactionStrategy(true).suspend();
            } catch (TransactionStrategyException e) {
                throw new ActionProcessingException("Error suspending transaction on service '" + serviceInvoker.getService() + "'", e);
            }
        }

        try {
            call.setFaultTo(null);
            call.setReplyTo(null);

            return serviceInvoker.deliverSync(message, timeout);
        } catch (Exception e) {
            // No real need to reset the EPRs because the pipeline has already captured them, but
            // lets do it anyway...
            call.setFaultTo(faultTo);
            call.setReplyTo(replyTo);

            if (failOnException) {
                throw new ActionProcessingException("Error delivering message to service '" + serviceInvoker.getService() + "'", e);
            } else {
                logger.debug("Exception during deliverSync().  Action configured to continue.", e);
                return message;
            }
        } finally {
            try {
                if(txObject != null) {
                    TransactionStrategy.getTransactionStrategy(true).resume(txObject);
                }
            } catch (TransactionStrategyException e) {
                throw new ActionProcessingException("Error resuming transaction on service '" + serviceInvoker.getService() + "'", e);
            }
        }
    }

    private boolean isTransactional() throws ActionProcessingException {
        try {
            TransactionStrategy txStrategy = TransactionStrategy.getTransactionStrategy(true);

            if (txStrategy == null) {
                return false;
            }

            if (txStrategy.getTransaction() != null) {
                if (txStrategy.isActive()) {
                    return true;
                } else {
                    throw new ActionProcessingException("Associated transaction is no longer active!");
                }
            }
        } catch (final TransactionStrategyException ex) {
            throw new ActionProcessingException("Could not determine transactionality.", ex);
        }

        return false;
    }
}
