/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.services.security;

import org.jboss.soa.esb.common.Configuration;
import org.jboss.soa.esb.services.security.SecurityServiceException;
import org.jboss.soa.esb.util.ClassUtil;

/**
 * Factory for creating {@link SecurityContextPropagator} instances.
 * Only contains factory methods.
 *
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 *
 */
public final class SecurityContextPropagatorFactory
{
    private SecurityContextPropagatorFactory() {}

    public static SecurityContextPropagator createFromConfig() throws SecurityServiceException
    {
        return create(Configuration.getSecurityContextPropagatorImplementationClass());
    }

    /**
     * Creates an instance of the specified className which must be an instance of {@link SecurityContextPropagator}.
     *
     * @param className The fully qualified name of the class to be used.
     * @return SecurityContextPropagator An instance of the passed in className of if className of null is className was null.
     * @throws SecurityServiceException If an exception occurs while trying to create the instance.
     */
    public static SecurityContextPropagator create(final String className) throws SecurityServiceException
    {
        if (className == null)
        {
            return null;
        }
        return createInstance(className);
    }

    private static SecurityContextPropagator createInstance(final String className) throws SecurityServiceException
    {
        try
        {
            final Class<?> propagator = ClassUtil.forName(className, SecurityContextPropagatorFactory.class);
            Object newInstance = propagator.newInstance();
            return (SecurityContextPropagator) newInstance;
        }
        catch (final ClassNotFoundException e)
        {
            throw new SecurityServiceException("SecurityContextPropagator implementation '" + className + "' not found", e);
        }
        catch (final InstantiationException e)
        {
            throw new SecurityServiceException("Instantiation Exception while trying to create a SecurityContextPropagator of type '" + className + "'", e);
        }
        catch (final IllegalAccessException e)
        {
            throw new SecurityServiceException("IllegalAccessException while trying to create a SecurityContextPropagator of type '" + className + "'", e);
        }
    }

}
