/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.services.security;

import java.io.IOException;
import java.io.Serializable;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.KeyGenerator;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SealedObject;
import javax.crypto.SecretKey;
import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.common.Configuration;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.services.security.SecurityServiceException;

/**
 * Util for sealing and unsealing object.
 *
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 */
public enum PrivateCryptoUtil
{
    INSTANCE;

    private Logger log;
    private Cipher cipher;
    private SecretKey secretKey;

    private PrivateCryptoUtil()
    {
        try
        {
            log = getLogger();
            initSecretKey();
        }
        catch (final Exception e)
        {
            throw new IllegalStateException(e.getMessage(), e);
        }
    }

    /**
     * Will create a Sealed object containing the passed in Serializable object.
     *
     * @param serializable The serializable object to be sealed.
     * @return SealedObject The resulting sealed object
     * @throws SecurityServiceException
     * @throws IOException
     * @throws IllegalBlockSizeException
     */
    public SealedObject sealObject(final Serializable serializable) throws SecurityServiceException
    {
        AssertArgument.isNotNull(serializable, "serializable");
        try
        {
            synchronized (cipher)
            {
                cipher.init(Cipher.ENCRYPT_MODE, secretKey);;
                return new SealedObject(serializable, cipher);
            }
        }
        catch (final InvalidKeyException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        catch (final IllegalBlockSizeException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        catch (final IOException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
    }

    /**
     * Unseals a previously sealed object.
     *
     * @param sealedObject The object to unseal
     * @return {@link Serializable} The unsealed Serializeable Object.
     * @throws SecurityServiceException
     */
    public Serializable unSealObject(SealedObject sealedObject) throws SecurityServiceException
    {
        AssertArgument.isNotNull(sealedObject, "sealedObject");
        try
        {
            synchronized (cipher)
            {
                cipher.init(Cipher.DECRYPT_MODE, secretKey);
                return (Serializable) sealedObject.getObject(cipher);
            }
        }
        catch (final InvalidKeyException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        catch (final IllegalBlockSizeException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        catch (final BadPaddingException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        catch (IOException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        catch (ClassNotFoundException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
    }

    /**
     * Generates a symmetric key used for sealing object.
     *
     * @throws SecurityServiceException
     */
    private void initSecretKey() throws SecurityServiceException
    {
        try
        {
            final String algorithm = Configuration.getSecurityServiceSealAlgorithm();
            if (algorithm == null)
            {
                throw new SecurityServiceException("'" + Environment.SECURITY_SERVICE_SEAL_ALGORITHM + "' has not been configured. Please set the algorithm to use in jbossesb-properties.xml");
            }

            final String keySizeStr = Configuration.getSecurityServiceSealKeySize();
            if (keySizeStr == null)
            {
                throw new SecurityServiceException("'" + Environment.SECURITY_SERVICE_SEAL_KEYSIZE + "' has not been configured. Please set the key size to use in jbossesb-properties.xml");
            }

            int keySize;
            try
            {
                keySize = Integer.parseInt(keySizeStr.trim());
            }
            catch(final NumberFormatException e)
            {
                throw new SecurityServiceException("'" + Environment.SECURITY_SERVICE_SEAL_KEYSIZE + "' must be a number. Please set the key size correctly in jbossesb-properties.xml");
            }

            log.debug("SealAlgorithm '" + algorithm + "' keySize '" + keySize + "'");

            final KeyGenerator kpg = KeyGenerator.getInstance(algorithm);
            kpg.init(keySize);
            secretKey = kpg.generateKey();
            cipher = Cipher.getInstance(algorithm);
        }
        catch (final NoSuchAlgorithmException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        catch (final NoSuchPaddingException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
    }

    private static Logger getLogger()
    {
        return Logger.getLogger(PrivateCryptoUtil.class);
    }
}
