/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.internal.soa.esb.remoting;

import org.apache.log4j.Logger;
import org.jboss.remoting.marshal.UnMarshaller;
import org.jboss.remoting.marshal.serializable.SerializableUnMarshaller;
import org.jboss.remoting.transport.http.HTTPMetadataConstants;
import org.jboss.soa.esb.http.HttpContentTypeUtil;

import java.io.*;
import java.util.List;
import java.util.Map;

/**
 * JBoss Remoting "http" unmarshaller.
 * <p/>
 * This is a direct copy of the HTTPUnMarshaller class from JBossRemoting.  It fixes
 * what we consider to be a bug in this class, which is the removal at new-line
 * characters (CR, LF) from the message through it's use of a BufferedReader.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class HttpUnmarshaller extends SerializableUnMarshaller {

    static final long serialVersionUID = 1085086661310576768L;

    private static final Logger log = Logger.getLogger(HttpUnmarshaller.class);

    /**
     * Will try to unmarshall data from inputstream.  Will try to convert to either an object
     * or a string.  If there is no data to read, will return null.
     *
     * @param inputStream
     * @param metadata
     * @param version
     * @return
     * @throws IOException
     * @throws ClassNotFoundException
     */
    public Object read(InputStream inputStream, Map metadata, int version) throws IOException, ClassNotFoundException {
        if (JBossRemotingUtil.receiveJavaObjectPayload(metadata)) {
            try {
                return super.read(inputStream, metadata, version);
            }
            catch (EOFException e) {
                return null;
            }
        }

        int bufferSize = 1024;
        byte[] byteBuffer = new byte[bufferSize];
        ByteArrayOutputStream byteOutputStream = new ByteArrayOutputStream();

        // check the metadat to see if is entry for content length
        if (metadata != null) {
            Object value = metadata.get("Content-Length");
            if (value == null) {
                value = metadata.get("content-length");
            }
            if (value != null) {
                if (value instanceof List) {
                    List valueList = (List) value;
                    if (valueList != null && valueList.size() > 0) {
                        value = valueList.get(0);
                    }
                }
                if (value instanceof String) {
                    try {
                        Integer.parseInt((String) value);
                    }
                    catch (NumberFormatException e) {
                        log.warn("Error converting Content-Length value (" + value + ") from metadata into int value.");
                    }
                } else {
                    log.warn("Can not get Content-Length from header for http unmarshalling.");
                }
            }
        }

        int pointer = 0;
        int amtRead = inputStream.read(byteBuffer);
        while (amtRead > 0) {
            byteOutputStream.write(byteBuffer, pointer, amtRead);
            amtRead = inputStream.read(byteBuffer);
        }

        byteOutputStream.flush();

        byte[] totalByteArray = byteOutputStream.toByteArray();

        if (totalByteArray.length == 0) {
            //nothing to read, so is null
            return null;
        }

        String contentType = JBossRemotingUtil.getContentType(metadata);
        if(!HttpContentTypeUtil.isTextMimetype(contentType)) {
            return totalByteArray ;
        }
        
        try {
            return new String(totalByteArray);
        }
        catch (Exception e) {
            log.error("Can not unmarshall inputstream.  Tried to unmarshall as both an object and string type.", e);
            throw new IOException("Can not unmarshall inputstream.");
        }
    }

    public UnMarshaller cloneUnMarshaller() throws CloneNotSupportedException {
        HttpUnmarshaller unmarshaller = new HttpUnmarshaller();
        unmarshaller.setClassLoader(this.customClassLoader);
        return unmarshaller;
    }

    private boolean isErrorReturn(Map metadata) {
        boolean isError = false;
        if (metadata != null) {
            // key of null will be the response (http type, response code, and response message)
            Object value = metadata.get(HTTPMetadataConstants.RESPONSE_CODE);
            if (value != null && value instanceof Integer) {
                int responseCode = ((Integer) value).intValue();
                if (responseCode > 400) {
                    isError = true;
                }
            }
        }
        return isError;
    }
}
