/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.notification;

import java.net.URI;

import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.mapping.ObjectMapper;
import org.jboss.soa.esb.message.mapping.ObjectMappingException;

/**
 * The property substituter applies two types of substitutions on a String - the first
 * is that it substitutes a message property contained in {property} for its value, 
 * the second being that it will use the ObjectMapper to substitute the value for a 
 * location within the message (example : body.order) if it is a String or a URI.
 * 
 * @author <a href="mailto:tcunning@redhat.com">tcunning@redhat.com</a>
 */
public class PropertySubstituter {
	public static String replaceArguments (String field, Message message) {
		String current = field;
		boolean foundReplacement = false;
		if (field == null) {
			return null;
		}
		
		String replaced = replaceArgument(field, message);
		
		while (!current.equals(replaced)) {
			current = replaced;
			foundReplacement = true;
			replaced = replaceArgument(current, message);
		}
		
		// See if the filename is an object in the message - if it is a 
		// String in the message, then we can use that value
		ObjectMapper om = new ObjectMapper();
		if (!foundReplacement) {
			try {
				Object obj = om.getObjectFromMessage(message, field);
				if (obj != null) {
					if (obj instanceof String) {
						replaced = (String) obj;
					} else if (obj instanceof URI) {
						replaced = obj.toString();
					}
				}
			} catch (ObjectMappingException e) {
			} catch (Exception e) {
			}
		}
		return replaced;
	}
	
	/**
	 * Look for arguments in the form '{prop.name}' and replace them with
	 * corresponding message properties.
	 * 
	 * @param value
	 * @param message
	 * @return String
	 */
	protected static String replaceArgument(String value, Message message) {
		int startIndex = value.indexOf('{');
		if (startIndex == -1) {
			return value;
		}
		int endIndex = value.indexOf('}');
		if (endIndex == -1) {
			return value;
		}
		String propName = value.substring(startIndex + 1, endIndex);
		Object propValue = message.getProperties().getProperty(propName);
		if (propValue == null) {
			return value;
		}
		return value.substring(0, startIndex) + propValue + value.substring(endIndex + 1);
	}
}
