package org.jboss.internal.soa.esb.message.format.xml;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.message.format.xml.marshal.MarshalUnmarshalManager;
import org.jboss.internal.soa.esb.message.format.xml.marshal.MarshalValueImpl;
import org.jboss.internal.soa.esb.util.Encoding;
import org.jboss.internal.soa.esb.util.stax.ElementContent;
import org.jboss.internal.soa.esb.util.stax.StreamHelper;
import org.jboss.internal.soa.esb.util.stax.TextElement;
import org.jboss.soa.esb.message.Body;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.util.Util;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

/**
 * A Body implementation that serializes to XML.
 * 
 * What this really should do is go through the Body contents and serialize the
 * basic types as XML types, thus improving loose coupling by reducing the
 * dependency on Java. It can do this by registering different Marshal/Unmarshal
 * classes.
 */

public class BodyImpl extends ElementContent implements Body
{
	public BodyImpl()
	{
	}
	
        /**
         * Construct a body from the input stream.
         * 
         * @param in The input stream.
         * @throws XMLStreamException For errors during parsing.
         */
        public BodyImpl(final XMLStreamReader in)
            throws XMLStreamException
        {
            parse(in) ;
        }

	public void add(Object value)
	{
		add(Body.DEFAULT_LOCATION, value);
	}

	public synchronized void add(String name, Object value)
	{
		AssertArgument.isNotNull(value, "value");
		AssertArgument.isNotNull(name, "name");

		_objects.put(name, new MarshalValueImpl(value));
	}

	public Object get()
	{
		return get(Body.DEFAULT_LOCATION);
	}

	public synchronized Object get(String name)
	{
		AssertArgument.isNotNull(name, "name");

		return unwrap(_objects.get(name));
	}

	public synchronized String[] getNames()
	{
		Set<String> keys = _objects.keySet();
                return keys.toArray(new String[0]) ;
	}
	
        public Enumeration<String> getEnumeratedNames()
        {
            return Collections.enumeration(Arrays.asList(getNames())) ;
        }

	public synchronized Object remove(String name)
	{
		return unwrap(_objects.remove(name));
	}

        /**
         * Write the child content of the element.
         * @param out The output stream.
         * @throws XMLStreamException For errors during output.
         */
        @Override
	protected synchronized void writeChildContent(XMLStreamWriter out)
	        throws XMLStreamException
	{
            /*
             * Keep separate byte array section for now, in case anyone is looking
             * for it within the XML (e.g., roll-your-own CBR).
             */
            final byte[] content = getByteArray() ;
            if (content != null)
            {
                final TextElement bytesContent = new TextElement(Encoding.encodeBytes(content)) ;
                StreamHelper.writeElement(out, XMLUtil.ESB_QNAME_BODY_BYTES_CONTENT, bytesContent) ;
            }
            
            // These should be handled by a proper element content
            for(Entry<String, MarshalValueImpl> entry: _objects.entrySet())
            {
                final String origContentURI = StreamHelper.writeStartElement(out, XMLUtil.ESB_QNAME_BODY_CONTENT) ;
                
                final TextElement key = new TextElement(Encoding.encodeBytes(entry.getKey().getBytes())) ;
                StreamHelper.writeElement(out, XMLUtil.ESB_QNAME_BODY_CONTENT_KEY, key) ;
                
                final String origValueURI = StreamHelper.writeStartElement(out, XMLUtil.ESB_QNAME_BODY_CONTENT_VALUE) ;
                if (!MarshalUnmarshalManager.getInstance().marshal(out, entry.getValue()))
                {
                    throw new XMLStreamException("Cannot pack object:" + entry.getKey()+" "+entry.getValue()) ;
                }
                StreamHelper.writeEndElement(out, XMLUtil.ESB_QNAME_BODY_CONTENT_VALUE.getPrefix(), origValueURI) ;
                
                StreamHelper.writeEndElement(out, XMLUtil.ESB_QNAME_BODY_CONTENT.getPrefix(), origContentURI) ;
            }
	}
	
        /**
         * Add the element.
         * @param in The current input stream.
         * @param elementName The qualified element name.
         */
        @Override
        protected void putElement(final XMLStreamReader in, final QName elementName)
            throws XMLStreamException
        {
            if (XMLUtil.ESB_QNAME_BODY_CONTENT.equals(elementName))
            {
                StreamHelper.checkNextStartTag(in, XMLUtil.ESB_QNAME_BODY_CONTENT_KEY) ;
                final TextElement keyElement = new TextElement(in) ;
                final String key = new String(Encoding.decodeToBytes(keyElement.getText())) ;

                StreamHelper.checkNextStartTag(in, XMLUtil.ESB_QNAME_BODY_CONTENT_VALUE) ;
                final MarshalValueImpl value = MarshalUnmarshalManager.getInstance().unmarshal(in) ;
                StreamHelper.checkParentFinished(in) ;
                StreamHelper.checkParentFinished(in) ;
                
                if (value == null)
                    throw new XMLStreamException("Cannot unpack object: " + key) ;
                else
                    _objects.put(key, value) ;
            }
            else if (XMLUtil.ESB_QNAME_BODY_BYTES_CONTENT.equals(elementName))
            {
                final TextElement textElement = new TextElement(in) ;
                final String value = textElement.getText();
                if (value != null)
                {
                    setByteArray(Encoding.decodeToBytes(value)) ;
                }
                else
                {
                    setByteArray(new byte[0]) ;
                }
            }
            else
            {
                throw new XMLStreamException("Unexpected element name: " + elementName) ;
            }
        }

	public void setByteArray(byte[] content)
	{
		add(BytesBody.BYTES_LOCATION, content);
	}

	public byte[] getByteArray()
	{
		return (byte[]) get(BytesBody.BYTES_LOCATION);
	}

	public void setContents(byte[] content)
	{
		setByteArray(content);
	}

	public byte[] getContents()
	{
		return getByteArray();
	}

	/**
	 * This method is not thread safe.
	 */
	public void replace(Body b)
	{
		if (b == null)
			throw new IllegalArgumentException();

		setByteArray(b.getByteArray());

		_objects = ((BodyImpl) b)._objects;
	}

	/**
	 * This method is not thread safe.
	 */
	public void merge(Body b)
	{
		if (b == null)
			throw new IllegalArgumentException();

		BodyImpl toMerge = null;

		if (b instanceof BodyImpl)
			toMerge = (BodyImpl) b;
		else
			throw new IllegalArgumentException();

		_objects.putAll(toMerge._objects);

		byte[] toAdd = b.getByteArray();

		if ((toAdd != null) && (toAdd.length > 0))
			setByteArray(toAdd);
	}

	public String toString()
	{
		StringBuilder toReturn = new StringBuilder("body: [ ");
		byte[] content = getByteArray();

		if (content != null)
			toReturn.append("byte[]: ").append(Util.format(new String(content))).append(", ");

		synchronized(this)
		{
			toReturn.append("objects: ").append(_objects);
		}

		return toReturn.append(" ]").toString() ;
	}

	// copy constructor (almost - doesn't copy, just references).

	protected BodyImpl(BodyImpl copy)
	{
		_objects = copy._objects;
	}

        private Object unwrap(final MarshalValueImpl value)
        {
            return (value == null ? null : value.getValue()) ;
        }
        
        protected void initialiseMap(final Map<String, Serializable> payload)
        {
            synchronized(this)
            {
                _objects.clear() ;
                for(Entry<String, Serializable> entry: payload.entrySet())
                {
                    _objects.put(entry.getKey(), new MarshalValueImpl(entry.getValue())) ;
                }
            }
        }
        
	private HashMap<String, MarshalValueImpl> _objects = new HashMap<String, MarshalValueImpl>();
}