/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.internal.soa.esb.couriers.helpers;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.addressing.eprs.FTPEpr;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.couriers.CourierUtil;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.util.RemoteFileSystem;
import org.jboss.soa.esb.util.RemoteFileSystemException;
import org.jboss.soa.esb.util.RemoteFileSystemFactory;

public class FtpFileHandler implements FileHandler
{
        FtpFileHandler(FTPEpr epr) throws CourierException
        {
                _epr = epr;

                final URI uri ;
                try
                {
                        uri = _epr.getURI();
                }
                catch (URISyntaxException e)
                {
                        throw new CourierException(e);
                }

                _server = uri.getHost();

                String[] sa = null;
                
                if (uri.getUserInfo() != null)
                        sa = uri.getUserInfo().split(":");
                
                if (sa == null)
                        sa = new String[]
                        { "", "" };
                _user = (sa.length < 1) ? "" : sa[0];
                _passwd = (sa.length < 2) ? "" : sa[1];

                _remoteDir = uri.getPath();
                
                final String tmpdir = System.getProperty("java.io.tmpdir") ;
                if ((_remoteDir == null) || (_remoteDir.equals("")))
                        _remoteDir = ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.FTP_REMOTEDIR, tmpdir);

                _port = uri.getPort();

                _localDir = ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.FTP_LOCALDIR, tmpdir);

                _isAscii = false;
                _isPassive = _epr.getPassive();
        }

        public boolean deleteFile(File file) throws CourierException
        {
                // TODO would be better to return false if the file did not exist and leave exceptions
                // for other error conditions (such as permission denied).
                
                final RemoteFileSystem handler = getHandler() ;
                try
                {
                        handler.deleteRemoteFile(file.getName());
                        return true;
                }
                catch (Exception e)
                {
                        e.printStackTrace();
                        
                        throw new CourierException(e);
                }
                finally
                {
                    handler.quit() ;
                }
        }

        public byte[] getFileContents(File file) throws CourierException
        {
                try
                {
                        /*
                         * For some reason the code only works on the file name at this
                         * point, even if a dir is passed in. So, if it's a dir
                         * we will ignore it and return null.
                         */

                        String name = file.getName();
                        
                        if (file.toString().length() > name.length())
                        {
                                _logger.debug("FtpFileHandler.getFileContents on "+file+" will ignore because of directory.");
                        
                                return null;
                        }
                        
                        final RemoteFileSystem handler = getHandler() ;
                        try
                        {
                            handler.downloadFile(name, name);
                        }
                        finally
                        {
                            handler.quit() ;
                        }
                        File local = new File(_localDir, name);
                        byte[] ba = CourierUtil.bytesFromLocalFile(local);
                        if (!local.delete())
                        {
                            _logger.warn("Could not delete file " + local.getCanonicalPath()) ;
                        }
                        return ba;
                }
                catch (Exception e)
                {
                        // TODO better error handling, e.g., what if someone
                        // tries to download a directory?
                        
                        throw new CourierException(e);
                }
        }

        // FileCourier will try to invoke the uploadFile(File) method using
        // reflection
        public void uploadFile(File file) throws CourierException
        {
                try
                {
                        String name = file.getName();
                        final RemoteFileSystem handler = getHandler() ;
                        try
                        {
                            handler.uploadFile(file, name);
                        }
                        finally
                        {
                            handler.quit();
                        }
                        if (!file.delete())
                        {
                            _logger.warn("Could not delete file " + file.getCanonicalPath()) ;
                        }
                }
                catch (Exception e)
                {
                        e.printStackTrace();
                        
                        throw new CourierException(e);
                }
        }

        // FileCourier will try to invoke the downloadFile(File) method using
        // reflection
        public File downloadFile(File file) throws CourierException
        {
                try
                {
                        String name = file.getName();
                        final RemoteFileSystem handler = getHandler() ;
                        try
                        {
                            handler.downloadFile(name, name);
                        }
                        finally
                        {
                            handler.quit() ;
                        }
                        return new File(_localDir,name);
                }
                catch (Exception e)
                {
                        throw new CourierException(e);
                }
        }

        public File[] getFileList() throws CourierException
        {
                String[] names = null;

                final RemoteFileSystem handler = getHandler() ;
                try
                {
                        names = handler.getFileListFromRemoteDir(_epr.getInputSuffix());
                }
                catch (Exception e)
                {
                        e.printStackTrace();
                        
                        throw new CourierException(e);
                }
                finally
                {
                    handler.quit() ;
                }

                if (null == names)
                        return null;
                File[] files = new File[names.length];
                int i = 0;
                for (String file : names)
                        if (null != file)
                                files[i++] = new File(file);
                return files;
        }

        public boolean renameFile(File from, File to) throws CourierException
        {
                final RemoteFileSystem handler = getHandler() ;
                try
                {
                        handler.remoteRename(from, to);

                        return true;
                }
                catch (IOException ex)  // file not found
                {
                        return false;
                }
                catch (Exception e)
                {
                        throw new CourierException(e);
                }
                finally
                {
                    handler.quit();
                }
        }

        // Because ftp connections are volatile, we are always getting a fresh
        // handle
        // TODO there's room for optimization here - not for GA though
        protected RemoteFileSystem getHandler() throws CourierException
        {
                try
                {
                        RemoteFileSystem rfs = RemoteFileSystemFactory.getRemoteFileSystem(
                                        _epr, true);
                        try
                        {
                                rfs.setRemoteDir(_remoteDir);
                        }
                        catch (Exception e)
                        {
                                throw new CourierException(e);
                        }

                        return rfs;
                }
                catch (RemoteFileSystemException e)
                {
                        throw new CourierException(e);
                }
        }

        private static final Logger _logger = Logger
                        .getLogger(FtpFileHandler.class);

        protected FTPEpr _epr;
        protected ConfigTree _tree;
        protected String _server;
        protected String _user;
        protected String _passwd;
        protected String _remoteDir;
        protected String _localDir;
        protected int _port;
        protected boolean _isAscii, _isPassive;
        
}
