/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

#ifndef DATATYPE_H
#define DATATYPE_H

// Qt includes
#include <qstringlist.h>

// WorKflow includes
#include "xmldescription.h"

// forward declarations
namespace WorKflow
{
  class LibraryDescription;
}

namespace WorKflow
{
  /**
   * @short This class represents a WorKflow datatype.
   *
   * @section overview Overview
   *
   * In WorKflow, each value is types. This class represents such a type. Types
   * can inherit from at most one base type. Like in OOP, if type @c B inherits
   * from type @c A, you can assign values of type @c B to values of type @c A,
   * but not neccessarily the other way around; to do this anyways, you need to
   * define and use conversions.
   *
   * Datatype inherits from XMLDescription, and as such provides the methods
   * id(), name() and description() to get the type's ID, the translated name
   * and a descriptive translated string. Since many values in WorKflow are
   * actually lists, the Datatype class provides the pluralName() method to get
   * a translated plural name, e.g. for type name @c "E-mail" the plural name
   * would be @c "E-mails".
   *
   * @section base Base Types and Conversions
   *
   * To get a type's base type, use baseType(). To see if the type inherits
   * (directly or through more than one step) from another type, use inherits().
   * To see if values of one type are convertible to another type, use
   * isConvertibleTo().
   *
   * @section enum Enumeration types.
   *
   * Other than types representing lists of values, it is very common that types
   * represent enumeration values. To see if a type represents enums, use the
   * isEnum() method. Enum values internally use non-translated strings, called
   * keys, which are used in Value objects. To the user, human-readable names
   * are presented, which represent the enum values . Use enumKeys() to get the
   * internal names of the enumeration values, and enumValues() to get
   * human-readable translated strings. The order of these items may change,
   * but to remain compatible with commands relying on an older list of enum
   * keys, enum items must not be removed. Also, keys must not be renamed.
   *
   * @section xml The XML file
   *
   * To define custom datatypes, you need to add a @c datatype tag to your
   * library's XML file.
   *
   * Example:
   *
   * @code
   * <datatype id="org.kde.public.imagefile" basetype="org.kde.public.file" >
   *     <name>Image File</name>
   *     <description>Local or remote image file</description>
   *     <pluralname>Image Files</pluralname>
   * </datatype>
   * @endcode
   *
   * To define enum datatypes, you need to add an @c enumitems tag as a child if
   * the @c datatype tag.
   *
   * Example:
   *
   * @code
   * <datatype id="org.kde.private.imagequality" >
   *     <name>Image Quality</name>
   *     <description>Image quality</description>
   *     <pluralname></pluralname>
   *     <enumitems>
   *         <item key="low" >Low</item>
   *         <item key="medium" >Medium</item>
   *         <item key="high" >High</item>
   *         <item key="best" >Best</item>
   *     </enumitems>
   * </datatype>
   * @endcode
   *
   * @section xmlref XML Tag Reference
   *
   * The following XML tags and attributes are recognized for datatypes:
   *
   * - @c datatype The datatype tag, required. The required @c id attribute defines
   *   the datatypes's ID. The id is accessible through the id() method.
   *   - @c name The datatype's name, in english. This will be translated to the
   *     user's current language. Use name() to get this translated text. Required.
   *   - @c pluralname The datatype's plural name, in english. This will be
   *     translated to the user's current language. Use pluralName() to get this
   *     translated text. Optional.
   *   - @c description A short help text, in english. This will be translated to
   *     the user's current languate. Use description() to get this text. Required.
   *   - @c enumitems Initiates the list of enum items. Optional.
   *     - @c item Defines an enumeration item. The @c key attribute is the
   *       internally used string to identify the enum item. The tag's text is the
   *       human-readable text representing the item. This text will be translated
   *       to the user's current language. Multiple @c item tags are possible.
   */
  class Datatype : public XMLDescription
  {
  public:
    /**
     * Constructor.
     */
    Datatype(LibraryDescription* parent);

    /**
     * Destructor.
     */
    ~Datatype();

    /**
     * Returns the datatype's plural name, e.g. "E-mails".
     * @return The datatype's plural name, i18n'ed.
     */
    QString pluralName() const;

    /**
     * Returns the datatype's base type, e.g. for "Image file" it would be
     * "File".
     * @return A pointer to this datatype's base type.
     */
    Datatype* baseType() const;

    /**
     * Sets the datatype's plural name.
     * @param plural The plural name.
     */
    void setPluralName(const QString& plural);

    /**
     * Returns the ID of the datatype's base type.
     * @return The basetype's ID.
     */
    QString baseTypeId() const;

    /**
     * Sets the datatype's base type.
     * @param id The base type's ID.
     */
    void setBaseTypeId(const QString& id);

    /**
     * Returns true, if a value of this datatype is convertible to a value of the
     * @p dest datatype. @see Conversion for how to implement custom conversions.
     * @param dest The datatype to convert to.
     * @return @c true, if a value of this type is convertible to @p dest.
     */
    bool isConvertibleTo(Datatype* dest);

    /**
     * Returns whether or not this datatype inherits from type @p base, either
     * directly or though a chain of base types.
     * @param base The type to check for if it's this type's base.
     * @return @c true, if this datatype inherits from @p base, @c false
     *   otherwise.
     */
    bool inherits(Datatype* base);

    /**
     * Returns whether or not this type is an enum.
     * @return @c true if the datatype is an enum type, @c false otherwise.
     */
    bool isEnum() const;

    /**
     * Returns the internal names of the enum items.
     * @return A list containing the enum values' keys.
     */
    QStringList enumKeys() const;

    /**
     * Returns the human-readable translated names which represent the enum
     * items.
     * @return A list containing the i18n'ed enum values' names.
     */
    QStringList enumNames() const;

    /**
     * Clears the list of enum keys and names. You should not need to call this
     * method, unless you have special needs for a dynamic enum type.
     */
    void clearEnumItems();

    /**
     * Adds an enum value to the list of enum values.
     * @param key The enum value's key.
     * @param name The enum value's human-readable name.
     */
    void addEnumItem(const QString& key, const QString& name);

    /**
     * Reimplemented from XMLDescription. Reads the datatypes's information from
     * the library's XML file.
     * @param e The datatype's XML tag node.
     */
    virtual void readXML(const QDomElement& e);

    /**
     * Reimplemented from XMLDescription. Writes the datatype's information to
     * the library's XML file.
     * @param doc The XML document object.
     * @param e The datatype's parent tag node (i.e. the library tag).
     */
    virtual void writeXML(QDomDocument& doc, QDomElement& e);

  protected:
    /**
     * Registers this datatype with the TypeManager class.
     */
    void registerType();

    /**
     * Removes this datatype from the list of registered types in the TypeManager
     * class.
     */
    void unregisterType();

  private:
    class Private;
    Private* d;
  };
}

#endif
