/*
 * kless.cpp. Part of the KDE project.
 *
 * Copyright (C) 1998 Matthias Ettrich
 *
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <kfiledialog.h>
#include "kless.h"
#include <qfile.h>
#include <qkeycode.h>
#include <qtextstream.h>
#include <qdragobject.h>

#include <kmessagebox.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kglobal.h>
#include <kmenubar.h>
#include <ktoolbar.h>
#include <kconfig.h>
#include <kaboutdata.h>
#include <kcmdlineargs.h>
#include <qfiledialog.h>

#include <kio/netaccess.h>

#define USE_KACCEL 1
#ifdef USE_KACCEL
#include <kaccel.h>
#else
#include <kstdaccel.h>
#endif

KLess::KLess() : KMainWindow(0, "kless")
{
  // create the main widget of kless: a QMultiLineEdit
  edit = new QMultiLineEdit(this);
  edit->setFont(QFont("courier"));
  
  // tell the KMainWindow that this is indeed the main widget
  setCentralWidget(edit);

  //---------------------------------------------------
  // Drag'n'Drop as implemented by Qt (XDnD)
  setAcceptDrops(TRUE);

  //---------------------------------------------------
  // build a menubar

  // a menubar consists of popup menus. So we need a
  // popup menu first
  QPopupMenu* p = new QPopupMenu;

#ifndef USE_KACCEL

  // the easy KDE-1.0 way. Simply use Qt's menus for acceleration.
  // KStdAccel gives you access to the user's current settings. You
  // could as well write "CTRL+Key_N" instead of a.openNew(), but that
  // would not be configurable, of course.

  KStdAccel a;
  p->insertItem(i18n("&New Window"), this, SLOT(slotNewWindow()), a.openNew());
  p->insertItem(i18n("&Load"), this, SLOT(slotLoadFile()), a.open() );
  p->insertSeparator();
  p->insertItem(i18n("&Quit"), this, SLOT(close()), a.quit());

#else

  // the new way in KDE post 1.1. KAccel is a very flexible class to handle
  // keyboard accelerations. These are completely configurable by the
  // user. Please see the kaccel documentation for further
  // details. The major advantage: With a single line of code calling
  // KKeyDialog::configureKeys(KAccel*) you can popup a keybinding
  // editor for your application.
  //
  // In kless we use KAccel only for some global defined standard
  // bindings.

  // create a kaccel object and bind the standard actions to our slots
  KAccel *a = new KAccel( this );
  a->connectItem(KStdAccel::New, this, SLOT( slotNewWindow() ) );
  a->connectItem(KStdAccel::Open, this, SLOT( slotLoadFile() ) );
  a->connectItem(KStdAccel::Quit, kapp, SLOT( quit() ) );

  // create the file menu. Tell kaccel to show the accelerators in the
  // menu as well. The advantage over doing that directly in Qt is,
  // that kaccels accelerators are internationalized.
  int id = p->insertItem(i18n("&New Window"),this, SLOT(slotNewWindow()));
  a->changeMenuAccel(p, id, KStdAccel::New );
  id = p->insertItem(i18n("&Open"),this, SLOT(slotLoadFile()));
  a->changeMenuAccel(p, id, KStdAccel::Open );
  p->insertSeparator();
  id = p->insertItem(i18n("&Quit"),this, SLOT(close()));
  a->changeMenuAccel(p, id, KStdAccel::Quit );

#endif

  // put the popupmenu as File-menu into the menu bar
  menuBar()->insertItem(i18n("&File"), p);
  menuBar()->insertSeparator();

  // we let KDE generate a nifty help menu
  p = helpMenu(i18n("kless --- the KDE text viewer :-) \n\n"
		    "(c) 1998 Matthias Ettrich\n"
		    "Demo Application for the K Desktop Environment."));
  menuBar()->insertItem(i18n("&Help"), p);

  //---------------------------------------------------
  // build a small toolbar

  // insert some buttons, the icons are from the standard KDE toolbar.
  toolBar()->insertButton(BarIcon("fileopen.xpm"),TOOLBAR_OPEN, true,i18n("Open File"));
  // Hint: toolBar() returns toolBar(0). If you want another toolbar, simply
  //       use toolBar(1), toolBar(2) and so on.

  // we connect the entire toolbar to one single slot. We could also
  // connect the single toolbar buttons to special slots in the
  // insertButton call above, for example:
  //   insertButton(BarIcon("exit.xpm"),TOOLBAR_EXIT,
  //                SIGNAL(clicked()), kapp, SLOT(quit()),
  //                true,"Exit")
  // This would be even better for this simple toolbar. We have chosen
  // the other design --- which is more flexible for bigger toolbars ---
  // for demonstration purposes in kless.
  connect(toolBar(), SIGNAL(clicked(int)), SLOT(slotToolbarClicked(int)));
}


void KLess::slotToolbarClicked(int item){
  switch (item) {
  case TOOLBAR_OPEN:
    slotLoadFile();
    break;
  }
}

void KLess::loadFile(const KURL & u){
  QString tgt;
  if (KIO::NetAccess::download(u, tgt)){
    setCaption( u.prettyURL() );
    loadFile(tgt);
    KIO::NetAccess::removeTempFile(tgt);
  }
}

void KLess::loadFile(const QString & filename){
  // load a file from a filename
  QFile f(filename);
  QString s;
  if ( f.open(IO_ReadOnly) ) {
    QTextStream t( &f );
    while ( !t.eof() ) {
      s += t.readLine() + "\n";
    }
    f.close();
  }
  // put the loaded text into the multiline edit
  edit->setText(s);
}

void KLess::dropEvent( QDropEvent *event){
  // the user dropped something on our window.
  // So we simply use KIO slaves to download the stuff wherever it is.
  // The following code respects Qt's DnD tutorial

  QImage im;
  QString txt, u;
  QStrList uri;

  if( QImageDrag::decode(event, im) )
    KMessageBox::sorry(this, i18n("This application accepts only URL drops"));

  if (QTextDrag::decode(event, txt)) {
    setCaption("The KDE text viewer");
    edit->setText(txt);
  }

  if(QUriDrag::decode(event, uri)) {

   u = uri.first();
   loadFile(u);
  }
}

void KLess::slotNewWindow(){
  // this slot is invoked from File->New_Winodw
  (new KLess)->show();
}

void KLess::slotLoadFile(){
  // this slot is invoked from File->Open

  // ask kfiledialog for a filename. We pass "this" to the function
  // to indicate the window manager that the file dialog belongs
  // to our application window.
  KURL u = KFileDialog::getOpenURL( QString::null, // initial dir
                                    QString::null, // filter
                                    this );

  if (!u.isEmpty()){ // no filename, user hit "Cancel" or pressed "Escape"
    loadFile( u );
  }
}


bool KLess::queryClose() {
  // this slot is invoked when the window is going to be closed
    switch ( KMessageBox::warningYesNoCancel( this, 
              i18n("Save changes to Document?<br>(Does not make sense, we know, but it is just a programming example :-)")))
    {
    case KMessageBox::Yes:
	// save document here. If saving fails, return FALSE;
	return TRUE;
    case KMessageBox::No:
	return TRUE;
    default: // cancel
	return FALSE;
    }
}

bool KLess::queryExit(){
  // this slot is invoked in addition when the *last* window is going
  // to be closed. We could do some final cleanup here.
    return TRUE; // accept
}


// Session management: save data
void KLess::saveProperties(KConfig* config){
  // This is provided just as an example.
  // It is generally not so good to save the raw contents of an application
  //  in its configuration file (as this example does).
  // It is preferable to save the contents in a file on the application's
  //  data zone and save an URL to it in the configuration resource.
  config->writeEntry("text", edit->text());
};

// Session management: read data again
void KLess::readProperties(KConfig* config){
  // See above
  edit->setText(config->readEntry("text"));
};

	// A Human readable description of your program
static const char *description=I18N_NOOP("KLess- Less is More");

	// The version
static const char *version="1.0";

	// The arguments your program accepts
static KCmdLineOptions options[] =
{
  { "+[file]", I18N_NOOP("File to open."), 0 },
  { 0, 0, 0}
};


int main(int argc, char* argv[]) {

  KAboutData aboutData( "kless", I18N_NOOP("KLess"),
    version, description, KAboutData::License_GPL,
    // License_GPL, License_LGPL, License_BSD, License_Artistic
    "(c) 1998, Matthias Ettrich"); // The Copyright
  aboutData.addAuthor("Matthias Ettrich",0, "ettrich@kde.org");
// void addAuthor(const char *name,
//   const char *task=0,
//   const char *emailAddress=0,
//   const char *webAddress=0);

  KCmdLineArgs::init( argc, argv, &aboutData );
  KCmdLineArgs::addCmdLineOptions( options );


  KApplication a;
  if (a.isRestored())
      RESTORE(KLess)
  else {
    // no session management: just create one window
    KLess* kless = new KLess;
    kless->show();

    KCmdLineArgs *args = KCmdLineArgs::parsedArgs();

    if (args->count())
    {// read kcmdlineargs.h for the full unabridged instructions
     // on using KCmdLineArgs, it's pretty confusing at first, but it works
      kless->loadFile( KURL(QDir::currentDirPath()+"/", args->arg(0) ) );
    }
    args->clear();
  }
  return a.exec();
}

#include "kless.moc"
