/* mg-graph-item.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "mg-graph-item.h"
#include <libmergeant/mg-ref-base.h>
#include <libmergeant/mg-xml-storage.h>
#include "marshal.h"

/* 
 * Main static functions 
 */
static void mg_graph_item_class_init (MgGraphItemClass *class);
static void mg_graph_item_init (MgGraphItem *graph);
static void mg_graph_item_dispose (GObject *object);
static void mg_graph_item_finalize (GObject *object);

static void mg_graph_item_set_property (GObject              *object,
					guint                 param_id,
					const GValue         *value,
					GParamSpec           *pspec);
static void mg_graph_item_get_property (GObject              *object,
					guint                 param_id,
					GValue               *value,
					GParamSpec           *pspec);

/* XML storage interface */
static void        mg_graph_item_xml_storage_init (MgXmlStorageIface *iface);
static xmlNodePtr  mg_graph_item_save_to_xml (MgXmlStorage *iface, GError **error);
static gboolean    mg_graph_item_load_from_xml (MgXmlStorage *iface, xmlNodePtr node, GError **error);


#ifdef debug
static void        mg_graph_item_dump                (MgGraphItem *graph, guint offset);
#endif

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* signals */
enum
{
	MOVED,
	LAST_SIGNAL
};

static gint mg_graph_item_signals[LAST_SIGNAL] = { 0 };

/* properties */
enum
{
	PROP_0,
	PROP_REF_OBJECT
};


struct _MgGraphItemPrivate
{
	MgRefBase    *ref_object;
	gdouble       x;
	gdouble       y;
};

/* module error */
GQuark mg_graph_item_error_quark (void)
{
	static GQuark quark;
	if (!quark)
		quark = g_quark_from_static_string ("mg_graph_item_error");
	return quark;
}


guint
mg_graph_item_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (MgGraphItemClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) mg_graph_item_class_init,
			NULL,
			NULL,
			sizeof (MgGraphItem),
			0,
			(GInstanceInitFunc) mg_graph_item_init
		};
		
		static const GInterfaceInfo xml_storage_info = {
                        (GInterfaceInitFunc) mg_graph_item_xml_storage_init,
                        NULL,
                        NULL
                };

		type = g_type_register_static (MG_BASE_TYPE, "MgGraphItem", &info, 0);
		g_type_add_interface_static (type, MG_XML_STORAGE_TYPE, &xml_storage_info);
	}
	return type;
}

static void
mg_graph_item_xml_storage_init (MgXmlStorageIface *iface)
{
        iface->get_xml_id = NULL;
        iface->save_to_xml = mg_graph_item_save_to_xml;
        iface->load_from_xml = mg_graph_item_load_from_xml;
}


static void
mg_graph_item_class_init (MgGraphItemClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);
	
	mg_graph_item_signals[MOVED] =
		g_signal_new ("moved",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (MgGraphItemClass, moved),
			      NULL, NULL,
			      marshal_VOID__VOID, G_TYPE_NONE, 0);

	object_class->dispose = mg_graph_item_dispose;
	object_class->finalize = mg_graph_item_finalize;

	/* Properties */
	object_class->set_property = mg_graph_item_set_property;
	object_class->get_property = mg_graph_item_get_property;

	g_object_class_install_property (object_class, PROP_REF_OBJECT,
					 g_param_spec_pointer ("ref_object", NULL, NULL, 
							       (G_PARAM_READABLE | G_PARAM_WRITABLE)));

	/* virtual functions */
#ifdef debug
        MG_BASE_CLASS (class)->dump = (void (*)(MgBase *, guint)) mg_graph_item_dump;
#endif
}

static void
mg_graph_item_init (MgGraphItem *graph)
{
	graph->priv = g_new0 (MgGraphItemPrivate, 1);
	graph->priv->ref_object = NULL;
	graph->priv->x = 0.;
	graph->priv->y = 0.;
}

/**
 * mg_graph_item_new
 * @conf: a #MgConf object
 * @ref_obj: a #MgBase object which the new item will reference, or %NULL.
 *
 * Creates a new #MgGraphItem object
 *
 * Returns: the newly created object
 */
GObject   *
mg_graph_item_new (MgConf *conf, MgBase *ref_obj)
{
	GObject *obj = NULL;
	MgGraphItem *item;

	g_return_val_if_fail (conf && IS_MG_CONF (conf), NULL);
	if (ref_obj)
		g_return_val_if_fail (IS_MG_BASE (ref_obj), NULL);

	obj = g_object_new (MG_GRAPH_ITEM_TYPE, "conf", conf, NULL);
	item = MG_GRAPH_ITEM (obj);

	item->priv->ref_object = MG_REF_BASE (mg_ref_base_new (conf));
	if (ref_obj)
		mg_ref_base_set_ref_object (item->priv->ref_object, ref_obj);

	return obj;
}

static void
mg_graph_item_dispose (GObject *object)
{
	MgGraphItem *graph;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_GRAPH_ITEM (object));

	graph = MG_GRAPH_ITEM (object);
	if (graph->priv) {		
		if (graph->priv->ref_object) {
			g_object_unref (G_OBJECT (graph->priv->ref_object));
                        graph->priv->ref_object = NULL;
                }
	}

	/* parent class */
	parent_class->dispose (object);
}


static void
mg_graph_item_finalize (GObject   * object)
{
	MgGraphItem *graph;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_GRAPH_ITEM (object));

	graph = MG_GRAPH_ITEM (object);
	if (graph->priv) {
		g_free (graph->priv);
		graph->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}


static void 
mg_graph_item_set_property (GObject              *object,
			   guint                 param_id,
			   const GValue         *value,
			   GParamSpec           *pspec)
{
	MgGraphItem *graph;
	gpointer ptr;

	graph = MG_GRAPH_ITEM (object);
	if (graph->priv) {
		switch (param_id) {
		case PROP_REF_OBJECT:
			g_assert (graph->priv->ref_object);
			if (graph->priv->ref_object) {
				ptr = g_value_get_pointer (value);
				mg_ref_base_set_ref_object (graph->priv->ref_object, ptr);
			}
			break;
		}
	}
}

static void
mg_graph_item_get_property (GObject              *object,
			   guint                 param_id,
			   GValue               *value,
			   GParamSpec           *pspec)
{
	MgGraphItem *graph;

	graph = MG_GRAPH_ITEM (object);
        if (graph->priv) {
                switch (param_id) {
                case PROP_REF_OBJECT:
			g_assert (graph->priv->ref_object);
			g_value_set_pointer (value, 
					     mg_ref_base_get_ref_object (graph->priv->ref_object));
                        break;
                }
        }
}

/**
 * mg_graph_item_set_position
 * @item: a #MgGraphItemItem object
 * @x:
 * @y:
 *
 * Sets the position to be remembered for @item.
 */
void
mg_graph_item_set_position (MgGraphItem *item, gdouble x, gdouble y)
{
	g_return_if_fail (item && IS_MG_GRAPH_ITEM (item));
	g_return_if_fail (item->priv);
	
	if ((item->priv->x == x) && (item->priv->y == y))
		return;

	item->priv->x = x;
	item->priv->y = y;
	
#ifdef debug_signal
	g_print (">> 'MOVED' from %s::%s()\n", __FILE__, __FUNCTION__);
#endif
	g_signal_emit (G_OBJECT (item), mg_graph_item_signals[MOVED], 0);
#ifdef debug_signal
	g_print ("<< 'MOVED' from %s::%s()\n", __FILE__, __FUNCTION__);
#endif
}

/**
 * mg_graph_item_get_position
 * @item: a #MgGraphItemItem object
 * @x: a place where to store the X part of the position, or %NULL
 * @y: a place where to store the Y part of the position, or %NULL
 *
 * Get @item's position.
 */
void
mg_graph_item_get_position (MgGraphItem *item, gdouble *x, gdouble *y)
{
	g_return_if_fail (item && IS_MG_GRAPH_ITEM (item));
	g_return_if_fail (item->priv);
	
	if (x)
		*x = item->priv->x;
	if (y)
		*y = item->priv->y;
}


/**
 * mg_graph_item_get_ref_object
 * @item: a #MgGraphItem object
 *
 * Get the referenced #MgBase object, if it exists.
 *
 * Returns: the referenced object, or %NULL
 */
MgBase *
mg_graph_item_get_ref_object (MgGraphItem *item)
{
	g_return_val_if_fail (item && IS_MG_GRAPH_ITEM (item), NULL);
	g_return_val_if_fail (item->priv, NULL);
	
	return mg_ref_base_get_ref_object (item->priv->ref_object);
}

#ifdef debug
static void
mg_graph_item_dump (MgGraphItem *item, guint offset)
{
	gchar *str;
	gint i;

	g_return_if_fail (item && IS_MG_GRAPH_ITEM (item));
	
        /* string for the offset */
        str = g_new0 (gchar, offset+1);
        for (i=0; i<offset; i++)
                str[i] = ' ';
        str[offset] = 0;

        /* dump */
        if (item->priv) {
		MgBase *base = mg_ref_base_get_ref_object (item->priv->ref_object);
		g_print ("%s" D_COL_H1 "MgGraphItem" D_COL_NOR " for %p at (%.3f, %.3f) ", str, base,
			 item->priv->x, item->priv->y);
		
	}
        else
                g_print ("%s" D_COL_ERR "Using finalized object %p" D_COL_NOR, str, item);
}
#endif

/* 
 * MgXmlStorage interface implementation
 */
static xmlNodePtr
mg_graph_item_save_to_xml (MgXmlStorage *iface, GError **error)
{
        xmlNodePtr node = NULL;
	MgGraphItem *item;
        gchar *str;
	MgBase *base;

        g_return_val_if_fail (iface && IS_MG_GRAPH_ITEM (iface), NULL);
        g_return_val_if_fail (MG_GRAPH_ITEM (iface)->priv, NULL);

        item = MG_GRAPH_ITEM (iface);

        node = xmlNewNode (NULL, "MG_GRAPH_ITEM");

	g_assert (item->priv->ref_object);
	base = mg_ref_base_get_ref_object (item->priv->ref_object);
	if (base) {
		str = mg_xml_storage_get_xml_id (MG_XML_STORAGE (base));
		xmlSetProp (node, "object", str);
		g_free (str);
	}
	
	str = g_strdup_printf ("%.3f", item->priv->x);
	xmlSetProp (node, "xpos", str);
	g_free (str);

	str = g_strdup_printf ("%.3f", item->priv->y);
	xmlSetProp (node, "ypos", str);
	g_free (str);

        return node;
}

static gboolean
mg_graph_item_load_from_xml (MgXmlStorage *iface, xmlNodePtr node, GError **error)
{
	MgGraphItem *item;
        gchar *prop;

        g_return_val_if_fail (iface && IS_MG_GRAPH_ITEM (iface), FALSE);
        g_return_val_if_fail (MG_GRAPH_ITEM (iface)->priv, FALSE);
        g_return_val_if_fail (node, FALSE);

	item = MG_GRAPH_ITEM (iface);

	if (strcmp (node->name, "MG_GRAPH_ITEM")) {
                g_set_error (error,
                             MG_GRAPH_ITEM_ERROR,
                             MG_GRAPH_ITEM_XML_LOAD_ERROR,
                             _("XML Tag is not <MG_GRAPH_ITEM>"));
                return FALSE;
        }

	prop = xmlGetProp (node, "object");
	if (prop) {
		g_assert (item->priv->ref_object);
		mg_ref_base_set_ref_name (item->priv->ref_object, 0/* FIXME */, REFERENCE_BY_XML_ID, prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "xpos");
	if (prop) {
		item->priv->x = atof (prop);
		g_free (prop);
	}

	prop = xmlGetProp (node, "ypos");
	if (prop) {
		item->priv->y = atof (prop);
		g_free (prop);
	}

        return TRUE;
}

