/* Mergeant
 *
 * Copyright (C) 1999 - 2003 Vivien Malerba
 * Copyright (C) 2002 - 2003 Rodrigo Moya
 *
 * Authors:
 *       Vivien Malerba <malerba@gnome-db.org>
 *       Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <bonobo/bonobo-ui-component.h>
#include <bonobo/bonobo-ui-util.h>
#include <bonobo/bonobo-window.h>
#include <glib/gspawn.h>
#include <gtk/gtkstatusbar.h>
#include <gtk/gtktable.h>
#include <libgnomeui/gnome-about.h>
#include <libgnomedb/libgnomedb.h>
#include <libgnomedb/gnome-db-login-dialog.h>
#include <libgnomedb/gnome-db-util.h>
#include <libmergeant/mg-dbms-update-viewer.h>
#include "workspace.h"
#include "workspace-window.h"
#include <libmergeant.h>

typedef struct {
	GtkWidget *window;
	GtkWidget *workspace;
	GtkWidget *status_bar;
	BonoboUIComponent *uic;

	/* the connection to the database */
	MgConf    *conf;
	MgServer  *server;

	GdaTransaction *pending_transaction;
} WorkspaceWindowPrivate;

static GList *opened_workspaces = NULL;

static void perform_dbms_update (WorkspaceWindowPrivate *priv);
static void stop_dbms_update_cb (GtkWidget *dialog, gint response, MgConf *conf);



static void
destroy_private_data_cb (WorkspaceWindowPrivate *priv)
{
	if (priv) {
		opened_workspaces = g_list_remove (opened_workspaces, priv);

		if (priv->conf) {
			mg_conf_save_xml (priv->conf, NULL); /* save before destruction */
			g_object_unref (priv->conf);
			priv->conf = NULL;
			priv->server = NULL;
		}

		if (priv->pending_transaction) {
			g_object_unref (priv->pending_transaction);
			priv->pending_transaction = NULL;
		}

		g_free (priv);

		if (!opened_workspaces)
			gnome_db_main_quit ();
	}
}

static void
on_file_new_workspace (BonoboUIComponent *uic, void *data, const char *path)
{
	workspace_window_new (NULL);
}

static void
on_file_copy_workspace (BonoboUIComponent *uic, void *data, const char *path)
{
	WorkspaceWindowPrivate *priv;

	priv = g_object_get_data (G_OBJECT (data), "Mergeant_WorkspaceWindowPrivate");
	workspace_window_new (priv->conf);
}

static void
on_file_close (BonoboUIComponent *uic, void *data, const char *path)
{
	GtkWidget *window = data;
	
	/* FIXME: check if window needs saving */
	gtk_widget_destroy (window);
}

static void
on_file_save (BonoboUIComponent *uic, void *data, const char *path)
{
	WorkspaceWindowPrivate *priv;
	GError *error = NULL;

	priv = g_object_get_data (G_OBJECT (data), "Mergeant_WorkspaceWindowPrivate");

	/* saving of MgConf */
	if (!mg_conf_save_xml (priv->conf, &error)) {
		g_warning ("Could not save: %s\n", error->message);
		g_error_free (error);
	}
}


static void
on_file_exit (BonoboUIComponent *uic, void *data, const char *path)
{
	while (opened_workspaces) {
		WorkspaceWindowPrivate *priv = opened_workspaces->data;

		on_file_close (uic, priv->window, path);
	}
}

static void
on_edit_delete (BonoboUIComponent *uic, void *data, const char *path)
{
}

static void
on_database_sync (BonoboUIComponent *uic, void *data, const char *path)
{
	WorkspaceWindowPrivate *priv;

	priv = g_object_get_data (G_OBJECT (data), "Mergeant_WorkspaceWindowPrivate");

	perform_dbms_update (priv);
	mg_conf_save_xml (priv->conf, NULL);
}

static void
on_database_begin (BonoboUIComponent *uic, void *data, const char *path)
{
}

static void
on_database_commit (BonoboUIComponent *uic, void *data, const char *path)
{
}

static void
on_database_rollback (BonoboUIComponent *uic, void *data, const char *path)
{
}

static void
on_tools_data_sources (BonoboUIComponent *uic, void *data, const char *path)
{
	char *argv[2];

	/* run gnome-database-properties config tool */
	argv[0] = (char *) "gnome-database-properties";
        argv[1] = NULL;
                                                                                  
	g_spawn_async (NULL, argv, NULL, G_SPAWN_SEARCH_PATH,
		       NULL, NULL, NULL, NULL);
}

static void
on_help_about (BonoboUIComponent *uic, void *data, const char *path)
{
	static GtkWidget *dialog = NULL;
                                                                                    
        if (dialog != NULL) {
                g_assert (GTK_WIDGET_REALIZED (dialog));
                gdk_window_show (dialog->window);
                gdk_window_raise (dialog->window);
        } else {
                GdkPixbuf *icon;
                const gchar *authors[] = {
                        "Vivien Malerba <malerba@gnome-db.org>",
                        "Fernando Martins <fmartins@hetnet.nl>",
                        "Rodrigo Moya <rodrigo@gnome-db.org>",
                        "Carlos Perello Marin <carlos@gnome-db.org>",
                        "Gonzalo Paniagua Javier <gonzalo@gnome-db.org>",
                        NULL
                };
                                                                                    
                const gchar *documenters[] = {
                        "Vivien Malerba <malerba@gnome-db.org>",
                        NULL
                };
		const gchar *translator_credits =
			"Ge'ez Frontier Foundation <locales@geez.org> Amharic translations\n" \
			"Mətin Əmirov <metin@karegen.com> Azerbaijani translations\n" \
			"Aleix Badia i Bosch <abadia@ica.es> Catalan translations\n" \
			"Miloslav Trmac <mitr@volny.cz> & Michal Bukovjan <bukm@centrum.cz> Czech translations\n" \
			"Ole Laursen <olau@hardworking.dk> Danish translations\n" \
                        "Gerhard Dieringer <DieringG@eba-haus.de> & Christian Neumair <chris@gnome-de.org> German translations\n" \
                        "Kostas Papadimas <pkst@gnome.org> Greek translations\n" \
                        "Alexander Winston <alexander.winston@comcast.net> English/Canada translations\n" \
                        "David Lodge <dave@cirt.net> English/British translations\n" \
                        "Francisco Javier F. Serrador <serrador@cvs.gnome.org> & Pablo del Campo <pablodc@bigfoot.com> Spanish translations\n" \
                        "Roozbeh Pournader <roozbeh@sharif.edu> Persian translations\n" \
                        "Christophe Merlet (RedFox) <redfox@redfoxcenter.org> French translation\n" \
                        "Alastair McKinstry <mckinstry@computer.org> Irish (gaeilge) translations\n" \
                        "Croatian language team <lokalizacija@linux.hr> Croatiann translations\n" \
                        "Mauro Colorio <linuxbox@interfree.it> & Alessio Dessì <alkex@inwind.it> Italian translations\n" \
                        "FSF-India <locale@gnu.org.in> Malayalam translations\n" \
                        "Norazah Abd Aziz  <azahaziz@yahoo.com> Malay translations\n" \
                        "Jan-Willem Harmanny <jwharmanny@hotmail.com> Dutch translations\n" \
                        "Kjartan Maraas <kmaraas@online.no> Norwegian translation\n" \
                        "GNOME PL Team <translators@gnome.pl> Polish translation\n" \
                        "Afonso Celso Medina <medina@maua.br> Português/Brasil translation\n" \
                        "Duarte Loreto <happyguy_pt@hotmail.com> Portuguese translations\n" \
                        "Valek Filippov <frob@df.ru> Russian translations\n" \
                        "Martin Lacko <lacko@host.sk> Slovak translations\n" \
                        "Goran Rakić <gox@devbase.net> Serbian translations\n" \
                        "Горан Ракић <gox@devbase.net> Serbian translations\n" \
                        "Christian Rose <menthos@menthos.com> Swedish translations\n" \
                        "Ali Pakkan <apakkan@hotmail.com> Turk translations\n" \
                        "Yuriy Syrota <yuri@renome.rovno.ua> Ukrainian translations\n" \
                        "Trinh Minh Thanh <tmthanh@linuxmail.org> Vietnamese translation\n" \
                        "Fan Zhang <a17841@bigpond.com> Simplified Chinese translations\n"
			; 
                                                                                    
                /* FIXME: use a real GError here */
                icon = gdk_pixbuf_new_from_file (MERGEANT_PIXMAPDIR "/mergeant.png", NULL);
                                                                                    
                dialog = gnome_about_new (_("Mergeant"), VERSION,
                                          "(C) 1999-2004 Vivien Malerba",
                                          _("A Database admin tool for any SQL database "
					    "accessible with the GNOME-DB module."),
                                          authors,
                                          documenters,
                                          translator_credits,
                                          icon);
		g_signal_connect (G_OBJECT (dialog), "destroy",
				  G_CALLBACK (gtk_widget_destroyed),
				  &dialog);
                gtk_widget_set_parent_window (GTK_WIDGET (dialog), GDK_WINDOW (data));
                gtk_widget_show (dialog);
                /* FIXME: do we have to unref() the pixbuf? */
        }
}

static BonoboUIVerb workspace_window_verbs[] = {
	BONOBO_UI_VERB ("FileNewWorkspace", on_file_new_workspace),
	BONOBO_UI_VERB ("FileCopyWorkspace", on_file_copy_workspace),
	BONOBO_UI_VERB ("FileClose", on_file_close),
	BONOBO_UI_VERB ("FileSave", on_file_save),
	BONOBO_UI_VERB ("FileExit", on_file_exit),
	BONOBO_UI_VERB ("EditDelete", on_edit_delete),
	BONOBO_UI_VERB ("DatabaseSync", on_database_sync),
	BONOBO_UI_VERB ("DatabaseBeginTransaction", on_database_begin),
	BONOBO_UI_VERB ("DatabaseCommitTransaction", on_database_commit),
	BONOBO_UI_VERB ("DatabaseRollbackTransaction", on_database_rollback),
	BONOBO_UI_VERB ("ToolsDataSources", on_tools_data_sources),
	BONOBO_UI_VERB ("HelpAbout", on_help_about),
	BONOBO_UI_VERB_END
};

static void
stop_dbms_update_cb (GtkWidget *dialog, gint response, MgConf *conf)
{
	mg_server_stop_update_dbms_data (mg_conf_get_server (conf));
	gtk_widget_destroy (dialog);
}

static void
perform_dbms_update (WorkspaceWindowPrivate *priv)
{
	GtkWidget *dialog, *view;
	GError *error;

	dialog = gtk_dialog_new_with_buttons (_("Metadata synchronisation"),
					      NULL,
					      GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);
	g_signal_connect (G_OBJECT (dialog), "response", G_CALLBACK (stop_dbms_update_cb), priv->conf);
	
	view = mg_dbms_update_viewer_new (priv->conf);
	gtk_widget_show (view);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), view, TRUE, TRUE, 6);
	gtk_widget_show (dialog);
	
	/* update types, functions and aggregates */
	error = NULL;
	if (!mg_server_update_dbms_data (priv->server, &error)) {
		if (error->code != MG_SERVER_META_DATA_UPDATE_USER_STOPPED) {
			gnome_db_show_error (_("Error updating server metadata:\n%s"), error->message);
			g_error_free (error);
		}
	}
	
	/* update database */
	error = NULL;
	if (!mg_database_update_dbms_data (mg_conf_get_database (priv->conf), &error)) {
		gnome_db_show_error (_("Error updating database metadata:\n%s"), error->message);
		g_error_free (error);
	}
	
	gtk_widget_destroy (dialog);
}

/**
 * workspace_window_new
 * @conf: a #MgConf object, or %NULL
 *
 * Creates a new workspace window. If @conf is not, %NULL
 * then it is reused for the new workspace.
 *
 * Returns:
 */
GtkWidget *
workspace_window_new (MgConf *conf)
{
	BonoboUIContainer *ui_container;
	WorkspaceWindowPrivate *priv;
	GtkWidget *dialog, *table;
	gchar *title;
	GError *error = NULL;
	MgServer *server = NULL;
	gchar *filename = NULL;
	gboolean file_loaded = FALSE;
	MgConf *workconf = conf;

	if (conf) {
		g_return_val_if_fail (IS_MG_CONF (conf), NULL);
		g_object_ref (G_OBJECT (conf));
	}
	
	while (! workconf) {
		const gchar *dsn;
		GList *list = opened_workspaces;

		/* dialog to choose the connection */
		dialog = gnome_db_login_dialog_new (_("Connect"));
		if (!gnome_db_login_dialog_run (GNOME_DB_LOGIN_DIALOG (dialog))) {
			gtk_widget_destroy (dialog);
			return NULL;
		}
		
		/* try to find another workspace window with the same DSN */
		dsn = gnome_db_login_dialog_get_dsn (GNOME_DB_LOGIN_DIALOG (dialog));
		while (list && !workconf) {
			gchar *dsn2;
			priv = (WorkspaceWindowPrivate *) list->data;
			dsn2 = mg_server_get_datasource (priv->server);
			if (!strcmp (dsn, dsn2)) {
				conf = priv->conf;
				workconf = priv->conf;
			}
			g_free (dsn2);
			
			list = g_list_next (list);
		}

		/* create a new workconf if none found */
		if (!workconf) {
			workconf = (MgConf *) mg_conf_new ();
			server = mg_conf_get_server (workconf);
			
			mg_server_set_datasource (server, dsn);
			filename = mg_conf_compute_xml_filename (workconf, dsn, NULL, NULL);
			mg_server_set_user_name (server, gnome_db_login_dialog_get_username (GNOME_DB_LOGIN_DIALOG (dialog)));
			mg_server_set_user_password (server, gnome_db_login_dialog_get_password (GNOME_DB_LOGIN_DIALOG (dialog)));
			
			/* try connection */
			error = NULL;
			if (!mg_server_open_connect (server, &error)) {
				gnome_db_show_error (error->message);
				g_error_free (error);
				g_object_unref (workconf);
				workconf = NULL;
			} 
		}

		gtk_widget_destroy (dialog);
	}

	/* create the private structure */
	priv = g_new0 (WorkspaceWindowPrivate, 1);
	priv->conf = workconf;
	priv->server = mg_conf_get_server (priv->conf);
	opened_workspaces = g_list_prepend (opened_workspaces, priv);

	g_object_set (G_OBJECT (priv->server), "with_functions", TRUE, NULL);

	/* create progress dialog to load DBMS data */
	if (!conf) {
		if (filename) {
			file_loaded = mg_conf_load_xml_file (workconf, filename, &error);
			if (! file_loaded) {
				if (error->code != MG_CONF_LOAD_FILE_NOT_EXIST_ERROR) 
					gnome_db_show_error (error->message);
				g_error_free (error);
			}
			mg_conf_set_xml_filename (workconf, filename);
		}
		
		if (!file_loaded) {
			perform_dbms_update (priv);
			
			if (filename)
				mg_conf_save_xml_file (workconf, filename, NULL);
		}
	}
	g_free (filename);

	/* create the window */
	priv->window = bonobo_window_new ("Mergeant", "Mergeant");
	g_object_set_data_full (G_OBJECT (priv->window), "Mergeant_WorkspaceWindowPrivate",
				priv, (GDestroyNotify) destroy_private_data_cb);
	gtk_window_set_icon_from_file (GTK_WINDOW (priv->window), MERGEANT_PIXMAPDIR "/mergeant.png", NULL);

	if (mg_server_get_user_name (priv->server)) 
		title = g_strdup_printf ("%s@%s - Mergeant", mg_server_get_user_name (priv->server),
					 mg_server_get_datasource (priv->server));
	else 
		title = g_strdup_printf ("%s - Mergeant", mg_server_get_datasource (priv->server));
	
	gtk_window_set_title (GTK_WINDOW (priv->window), title);
	g_free (title);
	gtk_widget_set_size_request (priv->window, 800, 700);

	ui_container = bonobo_window_get_ui_container (BONOBO_WINDOW (priv->window));

	priv->uic = bonobo_ui_component_new_default ();
	bonobo_ui_component_set_container (priv->uic, BONOBO_OBJREF (ui_container), NULL);

	/* create the workspace */
	table = gtk_table_new (2, 1, FALSE);
	gtk_widget_show (table);

	priv->workspace = workspace_new (priv->server);
	gtk_widget_show (priv->workspace);
	gtk_table_attach (GTK_TABLE (table), priv->workspace, 0, 1, 0, 1,
			  GTK_SHRINK | GTK_FILL | GTK_EXPAND,
			  GTK_SHRINK | GTK_FILL | GTK_EXPAND, 6, 6);

	priv->status_bar = gtk_statusbar_new ();
	gtk_widget_show (priv->status_bar);
	gtk_table_attach (GTK_TABLE (table), priv->status_bar, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 0, 0);

	bonobo_window_set_contents (BONOBO_WINDOW (priv->window), table);

	/* add the menu bar/toolbar */
	bonobo_ui_component_add_verb_list_with_data (priv->uic, workspace_window_verbs, priv->window);
	bonobo_ui_util_set_ui (priv->uic, MERGEANT_UIDIR, "mergeant.xml", "mergeant", NULL);

	/* display the window */
	workspace_window_sensitize_ui (priv->window);
	gtk_widget_show (priv->window);

	return priv->window;
}

void
workspace_window_sensitize_ui (GtkWidget *window)
{
}
