/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <src/SqliteConfigTimersStore.h>
#include <sqlite3.h>
#include <stdlib.h>
#include <string.h>
#include <gio/gio.h>
#include "src/ChannelList.h"
#include "src/Main.h"
#include "src/Channel.h"
#include "src/Utils.h"




struct _DVBSqliteConfigTimersStorePrivate {
	sqlite3_stmt* select_devices_statement;
	sqlite3_stmt* delete_group_statement;
	sqlite3_stmt* insert_group_statement;
	sqlite3_stmt* delete_device_statement;
	sqlite3_stmt* insert_device_statement;
	sqlite3_stmt* select_timers_statement;
	sqlite3_stmt* delete_timer_statement;
	sqlite3_stmt* insert_timer_statement;
	sqlite3_stmt* contains_group_statement;
	sqlite3_stmt* contains_timer_statement;
	sqlite3* db;
};

#define DVB_SQLITE_CONFIG_TIMERS_STORE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_SQLITE_CONFIG_TIMERS_STORE, DVBSqliteConfigTimersStorePrivate))
enum  {
	DVB_SQLITE_CONFIG_TIMERS_STORE_DUMMY_PROPERTY
};
#define DVB_SQLITE_CONFIG_TIMERS_STORE_CREATE_DEVICE_GROUPS "CREATE TABLE device_groups (\n        group_id INTEGER,\n        adapter_type INTEGER(1),\n        channels_file VARCHAR(255),\n        recordings_dir VARCHAR(255),\n        name VARCHAR(255),\n        PRIMARY KEY(group_id))"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_CREATE_DEVICES "CREATE TABLE devices (\n        group_id INTEGER,\n        adapter INTEGER,\n        frontend INTEGER,\n        PRIMARY KEY(adapter, frontend))"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_CREATE_TIMERS "CREATE TABLE timers (\n        timer_id INTEGER,\n        group_id INTEGER,\n        channel_sid INTEGER,\n        year INTEGER,\n        month INTEGER,\n        day INTEGER,\n        hour INTEGER,\n        minute INTEGER,\n        duration INTEGER,\n        PRIMARY KEY(timer_id))"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_SELECT_ALL_GROUPS "SELECT * FROM device_groups"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_SELECT_DEVICES "SELECT * FROM devices WHERE group_id=?"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_DELETE_GROUP "DELETE FROM device_groups WHERE group_id=?"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_INSERT_GROUP "INSERT INTO device_groups VALUES (?, ?, ?, ?, ?)"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_CONTAINS_GROUP "SELECT COUNT(*) FROM device_groups WHERE group_id=?"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_DELETE_DEVICE "DELETE FROM devices WHERE adapter=? AND frontend=?"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_INSERT_DEVICE "INSERT INTO devices VALUES (?, ?, ?)"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_SELECT_TIMERS "SELECT * FROM timers WHERE group_id=?"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_DELETE_TIMER "DELETE FROM timers WHERE timer_id=?"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_INSERT_TIMER "INSERT INTO timers VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)"
#define DVB_SQLITE_CONFIG_TIMERS_STORE_CONTAINS_TIMER "SELECT COUNT(*) FROM timers WHERE timer_id=?"
static GeeList* dvb_sqlite_config_timers_store_real_get_all_device_groups (DVBConfigStore* base);
static gboolean dvb_sqlite_config_timers_store_real_add_device_group (DVBConfigStore* base, DVBDeviceGroup* dev_group);
static gboolean dvb_sqlite_config_timers_store_real_remove_device_group (DVBConfigStore* base, DVBDeviceGroup* devgroup);
static gboolean dvb_sqlite_config_timers_store_real_add_device_to_group (DVBConfigStore* base, DVBDevice* dev, DVBDeviceGroup* devgroup);
static gboolean dvb_sqlite_config_timers_store_real_remove_device_from_group (DVBConfigStore* base, DVBDevice* dev, DVBDeviceGroup* devgroup);
static GeeList* dvb_sqlite_config_timers_store_real_get_all_timers_of_device_group (DVBTimersStore* base, DVBDeviceGroup* dev);
static gboolean dvb_sqlite_config_timers_store_real_add_timer_to_device_group (DVBTimersStore* base, DVBTimer* timer, DVBDeviceGroup* dev);
static gboolean dvb_sqlite_config_timers_store_real_remove_timer_from_device_group (DVBTimersStore* base, guint timer_id, DVBDeviceGroup* dev);
static void dvb_sqlite_config_timers_store_print_last_error (DVBSqliteConfigTimersStore* self);
static sqlite3* dvb_sqlite_config_timers_store_get_db_handler (void);
static GObject * dvb_sqlite_config_timers_store_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer dvb_sqlite_config_timers_store_parent_class = NULL;
static DVBConfigStoreIface* dvb_sqlite_config_timers_store_dvb_config_store_parent_iface = NULL;
static DVBTimersStoreIface* dvb_sqlite_config_timers_store_dvb_timers_store_parent_iface = NULL;
static void dvb_sqlite_config_timers_store_finalize (GObject* obj);



static GeeList* dvb_sqlite_config_timers_store_real_get_all_device_groups (DVBConfigStore* base) {
	DVBSqliteConfigTimersStore * self;
	GError * inner_error;
	GeeList* groups;
	sqlite3_stmt* statement;
	sqlite3_stmt* _tmp2;
	gint _tmp1;
	sqlite3_stmt* _tmp0;
	GeeList* _tmp8;
	self = (DVBSqliteConfigTimersStore*) base;
	inner_error = NULL;
	groups = (GeeList*) gee_array_list_new (DVB_TYPE_DEVICE_GROUP, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_equal);
	statement = NULL;
	_tmp2 = NULL;
	_tmp0 = NULL;
	if ((_tmp1 = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_SELECT_ALL_GROUPS, -1, &_tmp0, NULL), statement = (_tmp2 = _tmp0, (statement == NULL) ? NULL : (statement = (sqlite3_finalize (statement), NULL)), _tmp2), _tmp1) != SQLITE_OK) {
		GeeList* _tmp3;
		dvb_sqlite_config_timers_store_print_last_error (self);
		_tmp3 = NULL;
		return (_tmp3 = groups, (statement == NULL) ? NULL : (statement = (sqlite3_finalize (statement), NULL)), _tmp3);
	}
	while (sqlite3_step (statement) == SQLITE_ROW) {
		gint group_id;
		GeeList* devs;
		gint group_type;
		DVBAdapterType type;
		GFile* channels_file;
		DVBChannelList* channels;
		GFile* rec_dir;
		DVBDevice* ref_dev;
		DVBDeviceGroup* group;
		group_id = sqlite3_column_int (statement, 0);
		sqlite3_reset (self->priv->select_devices_statement);
		if (sqlite3_bind_int (self->priv->select_devices_statement, 1, group_id) != SQLITE_OK) {
			dvb_sqlite_config_timers_store_print_last_error (self);
			continue;
		}
		/* Get devices of group*/
		devs = (GeeList*) gee_array_list_new (DVB_TYPE_DEVICE, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_equal);
		while (sqlite3_step (self->priv->select_devices_statement) == SQLITE_ROW) {
			guint adapter;
			guint frontend;
			DVBDevice* _tmp4;
			adapter = (guint) sqlite3_column_int (self->priv->select_devices_statement, 1);
			frontend = (guint) sqlite3_column_int (self->priv->select_devices_statement, 2);
			/* Create new device*/
			_tmp4 = NULL;
			gee_collection_add ((GeeCollection*) devs, _tmp4 = dvb_device_new (adapter, frontend, TRUE));
			(_tmp4 == NULL) ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL));
		}
		/* No devices for this group*/
		if (gee_collection_get_size ((GeeCollection*) devs) == 0) {
			g_debug ("SqliteConfigTimersStore.vala:162: Group %d has no devices", group_id);
			(devs == NULL) ? NULL : (devs = (g_object_unref (devs), NULL));
			continue;
		}
		/* Get adapter type*/
		group_type = sqlite3_column_int (statement, 1);
		type = 0;
		switch (group_type) {
			case 1:
			{
				type = DVB_ADAPTER_TYPE_DVB_T;
				break;
			}
			case 2:
			{
				type = DVB_ADAPTER_TYPE_DVB_S;
				break;
			}
			case 3:
			{
				type = DVB_ADAPTER_TYPE_DVB_C;
				break;
			}
			default:
			{
				g_debug ("SqliteConfigTimersStore.vala:174: Group %d has unknown type %d", group_id, group_type);
				continue;
			}
		}
		/* Get channel list*/
		channels_file = g_file_new_for_path (sqlite3_column_text (statement, 2));
		channels = NULL;
		{
			DVBChannelList* _tmp5;
			DVBChannelList* _tmp6;
			_tmp5 = dvb_channel_list_restore_from_file (channels_file, type, (guint) group_id, &inner_error);
			if (inner_error != NULL) {
				goto __catch33_g_error;
				goto __finally33;
			}
			_tmp6 = NULL;
			channels = (_tmp6 = _tmp5, (channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL)), _tmp6);
		}
		goto __finally33;
		__catch33_g_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				g_warning ("SqliteConfigTimersStore.vala:187: Could not read channels: %s", e->message);
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
				(devs == NULL) ? NULL : (devs = (g_object_unref (devs), NULL));
				(channels_file == NULL) ? NULL : (channels_file = (g_object_unref (channels_file), NULL));
				(channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL));
				continue;
			}
		}
		__finally33:
		if (inner_error != NULL) {
			(devs == NULL) ? NULL : (devs = (g_object_unref (devs), NULL));
			(channels_file == NULL) ? NULL : (channels_file = (g_object_unref (channels_file), NULL));
			(channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL));
			(groups == NULL) ? NULL : (groups = (g_object_unref (groups), NULL));
			(statement == NULL) ? NULL : (statement = (sqlite3_finalize (statement), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
		rec_dir = g_file_new_for_path (sqlite3_column_text (statement, 3));
		/* Set reference device*/
		ref_dev = (DVBDevice*) gee_list_get (devs, 0);
		dvb_device_set_Channels (ref_dev, channels);
		dvb_device_set_RecordingsDirectory (ref_dev, rec_dir);
		/* Create device group*/
		group = dvb_device_group_new ((guint) group_id, ref_dev, !main_get_disable_epg_scanner ());
		dvb_device_group_set_Name (group, sqlite3_column_text (statement, 4));
		{
			gint i;
			i = 1;
			for (; i < gee_collection_get_size ((GeeCollection*) devs); i++) {
				DVBDevice* _tmp7;
				_tmp7 = NULL;
				dvb_device_group_add (group, _tmp7 = (DVBDevice*) gee_list_get (devs, i));
				(_tmp7 == NULL) ? NULL : (_tmp7 = (g_object_unref (_tmp7), NULL));
			}
		}
		gee_collection_add ((GeeCollection*) groups, group);
		(devs == NULL) ? NULL : (devs = (g_object_unref (devs), NULL));
		(channels_file == NULL) ? NULL : (channels_file = (g_object_unref (channels_file), NULL));
		(channels == NULL) ? NULL : (channels = (g_object_unref (channels), NULL));
		(rec_dir == NULL) ? NULL : (rec_dir = (g_object_unref (rec_dir), NULL));
		(ref_dev == NULL) ? NULL : (ref_dev = (g_object_unref (ref_dev), NULL));
		(group == NULL) ? NULL : (group = (g_object_unref (group), NULL));
	}
	_tmp8 = NULL;
	return (_tmp8 = groups, (statement == NULL) ? NULL : (statement = (sqlite3_finalize (statement), NULL)), _tmp8);
}


static gboolean dvb_sqlite_config_timers_store_real_add_device_group (DVBConfigStore* base, DVBDeviceGroup* dev_group) {
	DVBSqliteConfigTimersStore * self;
	char* channels;
	char* recdir;
	gboolean _tmp1;
	gboolean _tmp2;
	gboolean _tmp3;
	gboolean _tmp4;
	gboolean _tmp10;
	self = (DVBSqliteConfigTimersStore*) base;
	g_return_val_if_fail (dev_group != NULL, FALSE);
	if (dvb_sqlite_config_timers_store_contains_group (self, dvb_device_group_get_Id (dev_group))) {
		return FALSE;
	}
	channels = g_file_get_path (dvb_channel_list_get_channels_file (dvb_device_group_get_Channels (dev_group)));
	recdir = g_file_get_path (dvb_device_group_get_RecordingsDirectory (dev_group));
	sqlite3_reset (self->priv->insert_group_statement);
	_tmp1 = FALSE;
	_tmp2 = FALSE;
	_tmp3 = FALSE;
	_tmp4 = FALSE;
	if (sqlite3_bind_int (self->priv->insert_group_statement, 1, (gint) dvb_device_group_get_Id (dev_group)) != SQLITE_OK) {
		_tmp4 = TRUE;
	} else {
		_tmp4 = sqlite3_bind_int (self->priv->insert_group_statement, 2, (gint) dvb_device_group_get_Type (dev_group)) != SQLITE_OK;
	}
	if (_tmp4) {
		_tmp3 = TRUE;
	} else {
		const char* _tmp5;
		_tmp5 = NULL;
		_tmp3 = sqlite3_bind_text (self->priv->insert_group_statement, 3, (_tmp5 = channels, (_tmp5 == NULL) ? NULL : g_strdup (_tmp5)), -1, g_free) != SQLITE_OK;
	}
	if (_tmp3) {
		_tmp2 = TRUE;
	} else {
		const char* _tmp6;
		_tmp6 = NULL;
		_tmp2 = sqlite3_bind_text (self->priv->insert_group_statement, 4, (_tmp6 = recdir, (_tmp6 == NULL) ? NULL : g_strdup (_tmp6)), -1, g_free) != SQLITE_OK;
	}
	if (_tmp2) {
		_tmp1 = TRUE;
	} else {
		const char* _tmp7;
		_tmp7 = NULL;
		_tmp1 = sqlite3_bind_text (self->priv->insert_group_statement, 5, (_tmp7 = dvb_device_group_get_Name (dev_group), (_tmp7 == NULL) ? NULL : g_strdup (_tmp7)), -1, g_free) != SQLITE_OK;
	}
	if (_tmp1) {
		gboolean _tmp8;
		dvb_sqlite_config_timers_store_print_last_error (self);
		return (_tmp8 = FALSE, channels = (g_free (channels), NULL), recdir = (g_free (recdir), NULL), _tmp8);
	}
	if (sqlite3_step (self->priv->insert_group_statement) != SQLITE_DONE) {
		gboolean _tmp9;
		dvb_sqlite_config_timers_store_print_last_error (self);
		return (_tmp9 = FALSE, channels = (g_free (channels), NULL), recdir = (g_free (recdir), NULL), _tmp9);
	}
	{
		GeeIterator* _dev_it;
		_dev_it = gee_iterable_iterator ((GeeIterable*) dev_group);
		while (gee_iterator_next (_dev_it)) {
			DVBDevice* dev;
			dev = (DVBDevice*) gee_iterator_get (_dev_it);
			dvb_config_store_add_device_to_group ((DVBConfigStore*) self, dev, dev_group);
			(dev == NULL) ? NULL : (dev = (g_object_unref (dev), NULL));
		}
		(_dev_it == NULL) ? NULL : (_dev_it = (g_object_unref (_dev_it), NULL));
	}
	return (_tmp10 = TRUE, channels = (g_free (channels), NULL), recdir = (g_free (recdir), NULL), _tmp10);
}


static gboolean dvb_sqlite_config_timers_store_real_remove_device_group (DVBConfigStore* base, DVBDeviceGroup* devgroup) {
	DVBSqliteConfigTimersStore * self;
	self = (DVBSqliteConfigTimersStore*) base;
	g_return_val_if_fail (devgroup != NULL, FALSE);
	sqlite3_reset (self->priv->delete_group_statement);
	if (sqlite3_bind_int (self->priv->delete_group_statement, 1, (gint) dvb_device_group_get_Id (devgroup)) != SQLITE_OK) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		return FALSE;
	}
	if (sqlite3_step (self->priv->delete_group_statement) != SQLITE_DONE) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		return FALSE;
	}
	return TRUE;
}


gboolean dvb_sqlite_config_timers_store_contains_group (DVBSqliteConfigTimersStore* self, guint group_id) {
	gint c;
	g_return_val_if_fail (self != NULL, FALSE);
	sqlite3_reset (self->priv->contains_group_statement);
	if (sqlite3_bind_int (self->priv->contains_group_statement, 1, (gint) group_id) != SQLITE_OK) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		return FALSE;
	}
	c = 0;
	while (sqlite3_step (self->priv->contains_group_statement) == SQLITE_ROW) {
		c = sqlite3_column_int (self->priv->contains_group_statement, 0);
	}
	return c > 0;
}


static gboolean dvb_sqlite_config_timers_store_real_add_device_to_group (DVBConfigStore* base, DVBDevice* dev, DVBDeviceGroup* devgroup) {
	DVBSqliteConfigTimersStore * self;
	gboolean _tmp0;
	gboolean _tmp1;
	self = (DVBSqliteConfigTimersStore*) base;
	g_return_val_if_fail (dev != NULL, FALSE);
	g_return_val_if_fail (devgroup != NULL, FALSE);
	sqlite3_reset (self->priv->insert_device_statement);
	_tmp0 = FALSE;
	_tmp1 = FALSE;
	if (sqlite3_bind_int (self->priv->insert_device_statement, 1, (gint) dvb_device_group_get_Id (devgroup)) != SQLITE_OK) {
		_tmp1 = TRUE;
	} else {
		_tmp1 = sqlite3_bind_int (self->priv->insert_device_statement, 2, (gint) dvb_device_get_Adapter (dev)) != SQLITE_OK;
	}
	if (_tmp1) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = sqlite3_bind_int (self->priv->insert_device_statement, 3, (gint) dvb_device_get_Frontend (dev)) != SQLITE_OK;
	}
	if (_tmp0) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		return FALSE;
	}
	if (sqlite3_step (self->priv->insert_device_statement) != SQLITE_DONE) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		return FALSE;
	}
	return TRUE;
}


static gboolean dvb_sqlite_config_timers_store_real_remove_device_from_group (DVBConfigStore* base, DVBDevice* dev, DVBDeviceGroup* devgroup) {
	DVBSqliteConfigTimersStore * self;
	gboolean _tmp0;
	self = (DVBSqliteConfigTimersStore*) base;
	g_return_val_if_fail (dev != NULL, FALSE);
	g_return_val_if_fail (devgroup != NULL, FALSE);
	sqlite3_reset (self->priv->delete_device_statement);
	_tmp0 = FALSE;
	if (sqlite3_bind_int (self->priv->delete_device_statement, 1, (gint) dvb_device_get_Adapter (dev)) != SQLITE_OK) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = sqlite3_bind_int (self->priv->delete_device_statement, 2, (gint) dvb_device_get_Frontend (dev)) != SQLITE_OK;
	}
	if (_tmp0) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		return FALSE;
	}
	if (sqlite3_step (self->priv->delete_device_statement) != SQLITE_DONE) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		return FALSE;
	}
	return TRUE;
}


static GeeList* dvb_sqlite_config_timers_store_real_get_all_timers_of_device_group (DVBTimersStore* base, DVBDeviceGroup* dev) {
	DVBSqliteConfigTimersStore * self;
	GeeList* timers;
	self = (DVBSqliteConfigTimersStore*) base;
	g_return_val_if_fail (dev != NULL, NULL);
	timers = (GeeList*) gee_array_list_new (DVB_TYPE_TIMER, (GBoxedCopyFunc) g_object_ref, g_object_unref, g_direct_equal);
	sqlite3_reset (self->priv->select_timers_statement);
	if (sqlite3_bind_int (self->priv->select_timers_statement, 1, (gint) dvb_device_group_get_Id (dev)) != SQLITE_OK) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		return timers;
	}
	while (sqlite3_step (self->priv->select_timers_statement) == SQLITE_ROW) {
		guint tid;
		guint sid;
		guint duration;
		gint year;
		gint month;
		gint day;
		gint hour;
		gint minute;
		DVBChannel* channel;
		DVBTimer* _tmp1;
		tid = 0U;
		sid = 0U;
		duration = 0U;
		year = 0;
		month = 0;
		day = 0;
		hour = 0;
		minute = 0;
		tid = (guint) sqlite3_column_int (self->priv->select_timers_statement, 0);
		sid = (guint) sqlite3_column_int (self->priv->select_timers_statement, 2);
		year = sqlite3_column_int (self->priv->select_timers_statement, 3);
		month = sqlite3_column_int (self->priv->select_timers_statement, 4);
		day = sqlite3_column_int (self->priv->select_timers_statement, 5);
		hour = sqlite3_column_int (self->priv->select_timers_statement, 6);
		minute = sqlite3_column_int (self->priv->select_timers_statement, 7);
		duration = (guint) sqlite3_column_int (self->priv->select_timers_statement, 8);
		channel = dvb_channel_list_get_channel (dvb_device_group_get_Channels (dev), sid);
		_tmp1 = NULL;
		gee_collection_add ((GeeCollection*) timers, _tmp1 = dvb_timer_new ((guint32) tid, channel, year, month, day, hour, minute, duration, NULL));
		(_tmp1 == NULL) ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL));
		(channel == NULL) ? NULL : (channel = (g_object_unref (channel), NULL));
	}
	return timers;
}


static gboolean dvb_sqlite_config_timers_store_real_add_timer_to_device_group (DVBTimersStore* base, DVBTimer* timer, DVBDeviceGroup* dev) {
	DVBSqliteConfigTimersStore * self;
	gboolean _tmp1;
	gboolean _tmp2;
	gboolean _tmp3;
	gboolean _tmp4;
	gboolean _tmp5;
	gboolean _tmp6;
	gboolean _tmp7;
	gboolean _tmp8;
	self = (DVBSqliteConfigTimersStore*) base;
	g_return_val_if_fail (timer != NULL, FALSE);
	g_return_val_if_fail (dev != NULL, FALSE);
	if (dvb_sqlite_config_timers_store_contains_timer (self, (guint) dvb_timer_get_Id (timer))) {
		return FALSE;
	}
	sqlite3_reset (self->priv->insert_timer_statement);
	_tmp1 = FALSE;
	_tmp2 = FALSE;
	_tmp3 = FALSE;
	_tmp4 = FALSE;
	_tmp5 = FALSE;
	_tmp6 = FALSE;
	_tmp7 = FALSE;
	_tmp8 = FALSE;
	if (sqlite3_bind_int (self->priv->insert_timer_statement, 1, (gint) dvb_timer_get_Id (timer)) != SQLITE_OK) {
		_tmp8 = TRUE;
	} else {
		_tmp8 = sqlite3_bind_int (self->priv->insert_timer_statement, 2, (gint) dvb_device_group_get_Id (dev)) != SQLITE_OK;
	}
	if (_tmp8) {
		_tmp7 = TRUE;
	} else {
		_tmp7 = sqlite3_bind_int (self->priv->insert_timer_statement, 3, (gint) dvb_channel_get_Sid (dvb_timer_get_Channel (timer))) != SQLITE_OK;
	}
	if (_tmp7) {
		_tmp6 = TRUE;
	} else {
		_tmp6 = sqlite3_bind_int (self->priv->insert_timer_statement, 4, (gint) dvb_timer_get_Year (timer)) != SQLITE_OK;
	}
	if (_tmp6) {
		_tmp5 = TRUE;
	} else {
		_tmp5 = sqlite3_bind_int (self->priv->insert_timer_statement, 5, (gint) dvb_timer_get_Month (timer)) != SQLITE_OK;
	}
	if (_tmp5) {
		_tmp4 = TRUE;
	} else {
		_tmp4 = sqlite3_bind_int (self->priv->insert_timer_statement, 6, (gint) dvb_timer_get_Day (timer)) != SQLITE_OK;
	}
	if (_tmp4) {
		_tmp3 = TRUE;
	} else {
		_tmp3 = sqlite3_bind_int (self->priv->insert_timer_statement, 7, (gint) dvb_timer_get_Hour (timer)) != SQLITE_OK;
	}
	if (_tmp3) {
		_tmp2 = TRUE;
	} else {
		_tmp2 = sqlite3_bind_int (self->priv->insert_timer_statement, 8, (gint) dvb_timer_get_Minute (timer)) != SQLITE_OK;
	}
	if (_tmp2) {
		_tmp1 = TRUE;
	} else {
		_tmp1 = sqlite3_bind_int (self->priv->insert_timer_statement, 9, (gint) dvb_timer_get_Duration (timer)) != SQLITE_OK;
	}
	if (_tmp1) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		return FALSE;
	}
	if (sqlite3_step (self->priv->insert_timer_statement) != SQLITE_DONE) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		return FALSE;
	}
	return TRUE;
}


static gboolean dvb_sqlite_config_timers_store_real_remove_timer_from_device_group (DVBTimersStore* base, guint timer_id, DVBDeviceGroup* dev) {
	DVBSqliteConfigTimersStore * self;
	self = (DVBSqliteConfigTimersStore*) base;
	g_return_val_if_fail (dev != NULL, FALSE);
	sqlite3_reset (self->priv->delete_timer_statement);
	if (sqlite3_bind_int (self->priv->delete_timer_statement, 1, (gint) timer_id) != SQLITE_OK) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		return FALSE;
	}
	if (sqlite3_step (self->priv->delete_timer_statement) != SQLITE_DONE) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		return FALSE;
	}
	return TRUE;
}


gboolean dvb_sqlite_config_timers_store_contains_timer (DVBSqliteConfigTimersStore* self, guint timer_id) {
	gint c;
	g_return_val_if_fail (self != NULL, FALSE);
	sqlite3_reset (self->priv->contains_timer_statement);
	if (sqlite3_bind_int (self->priv->contains_timer_statement, 1, (gint) timer_id) != SQLITE_OK) {
		dvb_sqlite_config_timers_store_print_last_error (self);
		return FALSE;
	}
	c = 0;
	while (sqlite3_step (self->priv->contains_timer_statement) == SQLITE_ROW) {
		c = sqlite3_column_int (self->priv->contains_timer_statement, 0);
	}
	return c > 0;
}


static void dvb_sqlite_config_timers_store_print_last_error (DVBSqliteConfigTimersStore* self) {
	g_return_if_fail (self != NULL);
	g_critical ("SqliteConfigTimersStore.vala:391: SQLite error: %d, %s", sqlite3_errcode (self->priv->db), sqlite3_errmsg (self->priv->db));
}


static sqlite3* dvb_sqlite_config_timers_store_get_db_handler (void) {
	GError * inner_error;
	GFile* config_dir;
	GFile* config_cache;
	GFile* dbfile;
	gboolean create_tables;
	sqlite3* db;
	sqlite3* _tmp4;
	gint _tmp3;
	sqlite3* _tmp2;
	char* _tmp1;
	sqlite3* _tmp17;
	inner_error = NULL;
	config_dir = g_file_new_for_path (g_get_user_config_dir ());
	config_cache = g_file_get_child (config_dir, "gnome-dvb-daemon");
	dbfile = g_file_get_child (config_cache, "configtimers.sqlite3");
	if (!g_file_query_exists (config_cache, NULL)) {
		{
			dvb_utils_mkdirs (config_cache, &inner_error);
			if (inner_error != NULL) {
				goto __catch34_g_error;
				goto __finally34;
			}
		}
		goto __finally34;
		__catch34_g_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				sqlite3* _tmp0;
				g_critical ("SqliteConfigTimersStore.vala:405: Could not create directory: %s", e->message);
				_tmp0 = NULL;
				return (_tmp0 = NULL, (e == NULL) ? NULL : (e = (g_error_free (e), NULL)), (config_dir == NULL) ? NULL : (config_dir = (g_object_unref (config_dir), NULL)), (config_cache == NULL) ? NULL : (config_cache = (g_object_unref (config_cache), NULL)), (dbfile == NULL) ? NULL : (dbfile = (g_object_unref (dbfile), NULL)), _tmp0);
			}
		}
		__finally34:
		if (inner_error != NULL) {
			(config_dir == NULL) ? NULL : (config_dir = (g_object_unref (config_dir), NULL));
			(config_cache == NULL) ? NULL : (config_cache = (g_object_unref (config_cache), NULL));
			(dbfile == NULL) ? NULL : (dbfile = (g_object_unref (dbfile), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
			return NULL;
		}
	}
	create_tables = !g_file_query_exists (dbfile, NULL);
	db = NULL;
	_tmp4 = NULL;
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp3 = sqlite3_open (_tmp1 = g_file_get_path (dbfile), &_tmp2);
	db = (_tmp4 = _tmp2, (db == NULL) ? NULL : (db = (sqlite3_close (db), NULL)), _tmp4);
	_tmp3;
	_tmp1 = (g_free (_tmp1), NULL);
	if (create_tables) {
		char* errormsg;
		char* _tmp7;
		gint _tmp6;
		char* _tmp5;
		gint val;
		char* _tmp11;
		gint _tmp10;
		char* _tmp9;
		char* _tmp15;
		gint _tmp14;
		char* _tmp13;
		errormsg = NULL;
		_tmp7 = NULL;
		_tmp5 = NULL;
		val = (_tmp6 = sqlite3_exec (db, DVB_SQLITE_CONFIG_TIMERS_STORE_CREATE_DEVICE_GROUPS, NULL, NULL, &_tmp5), errormsg = (_tmp7 = _tmp5, errormsg = (g_free (errormsg), NULL), _tmp7), _tmp6);
		if (val != SQLITE_OK) {
			sqlite3* _tmp8;
			g_critical ("SqliteConfigTimersStore.vala:418: SQLite error: %s", errormsg);
			_tmp8 = NULL;
			return (_tmp8 = NULL, errormsg = (g_free (errormsg), NULL), (config_dir == NULL) ? NULL : (config_dir = (g_object_unref (config_dir), NULL)), (config_cache == NULL) ? NULL : (config_cache = (g_object_unref (config_cache), NULL)), (dbfile == NULL) ? NULL : (dbfile = (g_object_unref (dbfile), NULL)), (db == NULL) ? NULL : (db = (sqlite3_close (db), NULL)), _tmp8);
		}
		_tmp11 = NULL;
		_tmp9 = NULL;
		val = (_tmp10 = sqlite3_exec (db, DVB_SQLITE_CONFIG_TIMERS_STORE_CREATE_DEVICES, NULL, NULL, &_tmp9), errormsg = (_tmp11 = _tmp9, errormsg = (g_free (errormsg), NULL), _tmp11), _tmp10);
		if (val != SQLITE_OK) {
			sqlite3* _tmp12;
			g_critical ("SqliteConfigTimersStore.vala:424: SQLite error: %s", errormsg);
			_tmp12 = NULL;
			return (_tmp12 = NULL, errormsg = (g_free (errormsg), NULL), (config_dir == NULL) ? NULL : (config_dir = (g_object_unref (config_dir), NULL)), (config_cache == NULL) ? NULL : (config_cache = (g_object_unref (config_cache), NULL)), (dbfile == NULL) ? NULL : (dbfile = (g_object_unref (dbfile), NULL)), (db == NULL) ? NULL : (db = (sqlite3_close (db), NULL)), _tmp12);
		}
		_tmp15 = NULL;
		_tmp13 = NULL;
		val = (_tmp14 = sqlite3_exec (db, DVB_SQLITE_CONFIG_TIMERS_STORE_CREATE_TIMERS, NULL, NULL, &_tmp13), errormsg = (_tmp15 = _tmp13, errormsg = (g_free (errormsg), NULL), _tmp15), _tmp14);
		if (val != SQLITE_OK) {
			sqlite3* _tmp16;
			g_critical ("SqliteConfigTimersStore.vala:430: SQLite error: %s", errormsg);
			_tmp16 = NULL;
			return (_tmp16 = NULL, errormsg = (g_free (errormsg), NULL), (config_dir == NULL) ? NULL : (config_dir = (g_object_unref (config_dir), NULL)), (config_cache == NULL) ? NULL : (config_cache = (g_object_unref (config_cache), NULL)), (dbfile == NULL) ? NULL : (dbfile = (g_object_unref (dbfile), NULL)), (db == NULL) ? NULL : (db = (sqlite3_close (db), NULL)), _tmp16);
		}
		errormsg = (g_free (errormsg), NULL);
	}
	_tmp17 = NULL;
	return (_tmp17 = db, (config_dir == NULL) ? NULL : (config_dir = (g_object_unref (config_dir), NULL)), (config_cache == NULL) ? NULL : (config_cache = (g_object_unref (config_cache), NULL)), (dbfile == NULL) ? NULL : (dbfile = (g_object_unref (dbfile), NULL)), _tmp17);
}


DVBSqliteConfigTimersStore* dvb_sqlite_config_timers_store_construct (GType object_type) {
	DVBSqliteConfigTimersStore * self;
	self = g_object_newv (object_type, 0, NULL);
	return self;
}


DVBSqliteConfigTimersStore* dvb_sqlite_config_timers_store_new (void) {
	return dvb_sqlite_config_timers_store_construct (DVB_TYPE_SQLITE_CONFIG_TIMERS_STORE);
}


static GObject * dvb_sqlite_config_timers_store_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	DVBSqliteConfigTimersStoreClass * klass;
	GObjectClass * parent_class;
	DVBSqliteConfigTimersStore * self;
	klass = DVB_SQLITE_CONFIG_TIMERS_STORE_CLASS (g_type_class_peek (DVB_TYPE_SQLITE_CONFIG_TIMERS_STORE));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = DVB_SQLITE_CONFIG_TIMERS_STORE (obj);
	{
		sqlite3* _tmp0;
		sqlite3_stmt* _tmp3;
		gint _tmp2;
		sqlite3_stmt* _tmp1;
		sqlite3_stmt* _tmp6;
		gint _tmp5;
		sqlite3_stmt* _tmp4;
		sqlite3_stmt* _tmp9;
		gint _tmp8;
		sqlite3_stmt* _tmp7;
		sqlite3_stmt* _tmp12;
		gint _tmp11;
		sqlite3_stmt* _tmp10;
		sqlite3_stmt* _tmp15;
		gint _tmp14;
		sqlite3_stmt* _tmp13;
		sqlite3_stmt* _tmp18;
		gint _tmp17;
		sqlite3_stmt* _tmp16;
		sqlite3_stmt* _tmp21;
		gint _tmp20;
		sqlite3_stmt* _tmp19;
		sqlite3_stmt* _tmp24;
		gint _tmp23;
		sqlite3_stmt* _tmp22;
		sqlite3_stmt* _tmp27;
		gint _tmp26;
		sqlite3_stmt* _tmp25;
		sqlite3_stmt* _tmp30;
		gint _tmp29;
		sqlite3_stmt* _tmp28;
		_tmp0 = NULL;
		self->priv->db = (_tmp0 = dvb_sqlite_config_timers_store_get_db_handler (), (self->priv->db == NULL) ? NULL : (self->priv->db = (sqlite3_close (self->priv->db), NULL)), _tmp0);
		_tmp3 = NULL;
		_tmp1 = NULL;
		_tmp2 = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_SELECT_DEVICES, -1, &_tmp1, NULL);
		self->priv->select_devices_statement = (_tmp3 = _tmp1, (self->priv->select_devices_statement == NULL) ? NULL : (self->priv->select_devices_statement = (sqlite3_finalize (self->priv->select_devices_statement), NULL)), _tmp3);
		_tmp2;
		_tmp6 = NULL;
		_tmp4 = NULL;
		_tmp5 = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_DELETE_GROUP, -1, &_tmp4, NULL);
		self->priv->delete_group_statement = (_tmp6 = _tmp4, (self->priv->delete_group_statement == NULL) ? NULL : (self->priv->delete_group_statement = (sqlite3_finalize (self->priv->delete_group_statement), NULL)), _tmp6);
		_tmp5;
		_tmp9 = NULL;
		_tmp7 = NULL;
		_tmp8 = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_INSERT_GROUP, -1, &_tmp7, NULL);
		self->priv->insert_group_statement = (_tmp9 = _tmp7, (self->priv->insert_group_statement == NULL) ? NULL : (self->priv->insert_group_statement = (sqlite3_finalize (self->priv->insert_group_statement), NULL)), _tmp9);
		_tmp8;
		_tmp12 = NULL;
		_tmp10 = NULL;
		_tmp11 = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_DELETE_DEVICE, -1, &_tmp10, NULL);
		self->priv->delete_device_statement = (_tmp12 = _tmp10, (self->priv->delete_device_statement == NULL) ? NULL : (self->priv->delete_device_statement = (sqlite3_finalize (self->priv->delete_device_statement), NULL)), _tmp12);
		_tmp11;
		_tmp15 = NULL;
		_tmp13 = NULL;
		_tmp14 = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_INSERT_DEVICE, -1, &_tmp13, NULL);
		self->priv->insert_device_statement = (_tmp15 = _tmp13, (self->priv->insert_device_statement == NULL) ? NULL : (self->priv->insert_device_statement = (sqlite3_finalize (self->priv->insert_device_statement), NULL)), _tmp15);
		_tmp14;
		_tmp18 = NULL;
		_tmp16 = NULL;
		_tmp17 = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_SELECT_TIMERS, -1, &_tmp16, NULL);
		self->priv->select_timers_statement = (_tmp18 = _tmp16, (self->priv->select_timers_statement == NULL) ? NULL : (self->priv->select_timers_statement = (sqlite3_finalize (self->priv->select_timers_statement), NULL)), _tmp18);
		_tmp17;
		_tmp21 = NULL;
		_tmp19 = NULL;
		_tmp20 = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_DELETE_TIMER, -1, &_tmp19, NULL);
		self->priv->delete_timer_statement = (_tmp21 = _tmp19, (self->priv->delete_timer_statement == NULL) ? NULL : (self->priv->delete_timer_statement = (sqlite3_finalize (self->priv->delete_timer_statement), NULL)), _tmp21);
		_tmp20;
		_tmp24 = NULL;
		_tmp22 = NULL;
		_tmp23 = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_INSERT_TIMER, -1, &_tmp22, NULL);
		self->priv->insert_timer_statement = (_tmp24 = _tmp22, (self->priv->insert_timer_statement == NULL) ? NULL : (self->priv->insert_timer_statement = (sqlite3_finalize (self->priv->insert_timer_statement), NULL)), _tmp24);
		_tmp23;
		_tmp27 = NULL;
		_tmp25 = NULL;
		_tmp26 = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_CONTAINS_GROUP, -1, &_tmp25, NULL);
		self->priv->contains_group_statement = (_tmp27 = _tmp25, (self->priv->contains_group_statement == NULL) ? NULL : (self->priv->contains_group_statement = (sqlite3_finalize (self->priv->contains_group_statement), NULL)), _tmp27);
		_tmp26;
		_tmp30 = NULL;
		_tmp28 = NULL;
		_tmp29 = sqlite3_prepare (self->priv->db, DVB_SQLITE_CONFIG_TIMERS_STORE_CONTAINS_TIMER, -1, &_tmp28, NULL);
		self->priv->contains_timer_statement = (_tmp30 = _tmp28, (self->priv->contains_timer_statement == NULL) ? NULL : (self->priv->contains_timer_statement = (sqlite3_finalize (self->priv->contains_timer_statement), NULL)), _tmp30);
		_tmp29;
	}
	return obj;
}


static void dvb_sqlite_config_timers_store_class_init (DVBSqliteConfigTimersStoreClass * klass) {
	dvb_sqlite_config_timers_store_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBSqliteConfigTimersStorePrivate));
	G_OBJECT_CLASS (klass)->constructor = dvb_sqlite_config_timers_store_constructor;
	G_OBJECT_CLASS (klass)->finalize = dvb_sqlite_config_timers_store_finalize;
}


static void dvb_sqlite_config_timers_store_dvb_config_store_interface_init (DVBConfigStoreIface * iface) {
	dvb_sqlite_config_timers_store_dvb_config_store_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_all_device_groups = dvb_sqlite_config_timers_store_real_get_all_device_groups;
	iface->add_device_group = dvb_sqlite_config_timers_store_real_add_device_group;
	iface->remove_device_group = dvb_sqlite_config_timers_store_real_remove_device_group;
	iface->add_device_to_group = dvb_sqlite_config_timers_store_real_add_device_to_group;
	iface->remove_device_from_group = dvb_sqlite_config_timers_store_real_remove_device_from_group;
}


static void dvb_sqlite_config_timers_store_dvb_timers_store_interface_init (DVBTimersStoreIface * iface) {
	dvb_sqlite_config_timers_store_dvb_timers_store_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_all_timers_of_device_group = dvb_sqlite_config_timers_store_real_get_all_timers_of_device_group;
	iface->add_timer_to_device_group = dvb_sqlite_config_timers_store_real_add_timer_to_device_group;
	iface->remove_timer_from_device_group = dvb_sqlite_config_timers_store_real_remove_timer_from_device_group;
}


static void dvb_sqlite_config_timers_store_instance_init (DVBSqliteConfigTimersStore * self) {
	self->priv = DVB_SQLITE_CONFIG_TIMERS_STORE_GET_PRIVATE (self);
}


static void dvb_sqlite_config_timers_store_finalize (GObject* obj) {
	DVBSqliteConfigTimersStore * self;
	self = DVB_SQLITE_CONFIG_TIMERS_STORE (obj);
	(self->priv->select_devices_statement == NULL) ? NULL : (self->priv->select_devices_statement = (sqlite3_finalize (self->priv->select_devices_statement), NULL));
	(self->priv->delete_group_statement == NULL) ? NULL : (self->priv->delete_group_statement = (sqlite3_finalize (self->priv->delete_group_statement), NULL));
	(self->priv->insert_group_statement == NULL) ? NULL : (self->priv->insert_group_statement = (sqlite3_finalize (self->priv->insert_group_statement), NULL));
	(self->priv->delete_device_statement == NULL) ? NULL : (self->priv->delete_device_statement = (sqlite3_finalize (self->priv->delete_device_statement), NULL));
	(self->priv->insert_device_statement == NULL) ? NULL : (self->priv->insert_device_statement = (sqlite3_finalize (self->priv->insert_device_statement), NULL));
	(self->priv->select_timers_statement == NULL) ? NULL : (self->priv->select_timers_statement = (sqlite3_finalize (self->priv->select_timers_statement), NULL));
	(self->priv->delete_timer_statement == NULL) ? NULL : (self->priv->delete_timer_statement = (sqlite3_finalize (self->priv->delete_timer_statement), NULL));
	(self->priv->insert_timer_statement == NULL) ? NULL : (self->priv->insert_timer_statement = (sqlite3_finalize (self->priv->insert_timer_statement), NULL));
	(self->priv->contains_group_statement == NULL) ? NULL : (self->priv->contains_group_statement = (sqlite3_finalize (self->priv->contains_group_statement), NULL));
	(self->priv->contains_timer_statement == NULL) ? NULL : (self->priv->contains_timer_statement = (sqlite3_finalize (self->priv->contains_timer_statement), NULL));
	(self->priv->db == NULL) ? NULL : (self->priv->db = (sqlite3_close (self->priv->db), NULL));
	G_OBJECT_CLASS (dvb_sqlite_config_timers_store_parent_class)->finalize (obj);
}


GType dvb_sqlite_config_timers_store_get_type (void) {
	static GType dvb_sqlite_config_timers_store_type_id = 0;
	if (dvb_sqlite_config_timers_store_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBSqliteConfigTimersStoreClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_sqlite_config_timers_store_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBSqliteConfigTimersStore), 0, (GInstanceInitFunc) dvb_sqlite_config_timers_store_instance_init, NULL };
		static const GInterfaceInfo dvb_config_store_info = { (GInterfaceInitFunc) dvb_sqlite_config_timers_store_dvb_config_store_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo dvb_timers_store_info = { (GInterfaceInitFunc) dvb_sqlite_config_timers_store_dvb_timers_store_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		dvb_sqlite_config_timers_store_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBSqliteConfigTimersStore", &g_define_type_info, 0);
		g_type_add_interface_static (dvb_sqlite_config_timers_store_type_id, DVB_TYPE_CONFIG_STORE, &dvb_config_store_info);
		g_type_add_interface_static (dvb_sqlite_config_timers_store_type_id, DVB_TYPE_TIMERS_STORE, &dvb_timers_store_info);
	}
	return dvb_sqlite_config_timers_store_type_id;
}




